<?php
$pageTitle = 'Track Order - OrderFood Portal';
require_once 'config.php';

$order = null;
$error = null;

// Get order by order number
if (isset($_GET['order'])) {
    $orderNumber = trim($_GET['order']);
    
    if (!empty($orderNumber)) {
        $db = getDB();
        $stmt = $db->prepare("SELECT * FROM orders WHERE order_number = ?");
        $stmt->execute([$orderNumber]);
        $order = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$order) {
            $error = "Order not found. Please check your order number.";
        } else {
            // Get order items
            $itemsStmt = $db->prepare("SELECT oi.*, m.name as item_name 
                                       FROM order_items oi 
                                       LEFT JOIN menu_items m ON oi.menu_item_id = m.id 
                                       WHERE oi.order_id = ?");
            $itemsStmt->execute([$order['id']]);
            $order['items'] = $itemsStmt->fetchAll(PDO::FETCH_ASSOC);
        }
    }
}

// Handle form submission for searching
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['order_number'])) {
    $searchNumber = trim($_POST['order_number']);
    if (!empty($searchNumber)) {
        header('Location: track-order.php?order=' . urlencode($searchNumber));
        exit;
    }
}

// Define status steps
$statusSteps = [
    'pending' => 1,
    'confirmed' => 2,
    'preparing' => 3,
    'ready' => 4,
    'out_for_delivery' => 5,
    'delivered' => 6
];

$currentStep = $order ? ($statusSteps[$order['order_status']] ?? 1) : 0;

require_once 'includes/header.php';
?>

<!-- Page Header -->
<section class="page-header">
    <div class="container">
        <h1 data-aos="fade-up">Track Your Order</h1>
        <nav aria-label="breadcrumb" data-aos="fade-up" data-aos-delay="100">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="index.php">Home</a></li>
                <li class="breadcrumb-item active">Track Order</li>
            </ol>
        </nav>
    </div>
</section>

<!-- Track Order Section -->
<section class="py-5">
    <div class="container">
        <?php if (!$order): ?>
        <!-- Search Form -->
        <div class="row justify-content-center">
            <div class="col-lg-6">
                <div class="card">
                    <div class="card-body p-4 text-center">
                        <div class="mb-4">
                            <i class="bi bi-search" style="font-size: 48px; color: var(--primary-color);"></i>
                        </div>
                        <h4 class="mb-3">Enter Your Order Number</h4>
                        <p class="text-muted mb-4">Track your order status in real-time</p>
                        
                        <?php if ($error): ?>
                        <div class="alert alert-danger">
                            <i class="bi bi-exclamation-circle me-2"></i><?= htmlspecialchars($error) ?>
                        </div>
                        <?php endif; ?>
                        
                        <form method="POST" action="">
                            <div class="input-group mb-3">
                                <span class="input-group-text"><i class="bi bi-hash"></i></span>
                                <input type="text" class="form-control form-control-lg" name="order_number" 
                                       placeholder="e.g., ORD202512030001" required>
                            </div>
                            <button type="submit" class="btn btn-primary btn-lg w-100">
                                <i class="bi bi-search me-2"></i>Track Order
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
        <?php else: ?>
        <!-- Order Details -->
        <div class="row">
            <div class="col-lg-8 mx-auto">
                <!-- Order Info Card -->
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white">
                        <div class="d-flex justify-content-between align-items-center">
                            <h5 class="mb-0">
                                <i class="bi bi-receipt me-2"></i>Order #<?= htmlspecialchars($order['order_number']) ?>
                            </h5>
                            <span class="badge bg-light text-dark">
                                <?= ucfirst(str_replace('_', ' ', $order['order_status'])) ?>
                            </span>
                        </div>
                    </div>
                    <div class="card-body">
                        <div class="row mb-4">
                            <div class="col-sm-6 mb-3">
                                <small class="text-muted d-block">Customer Name</small>
                                <strong><?= htmlspecialchars($order['customer_name']) ?></strong>
                            </div>
                            <div class="col-sm-6 mb-3">
                                <small class="text-muted d-block">Phone Number</small>
                                <strong><?= htmlspecialchars($order['customer_phone']) ?></strong>
                            </div>
                            <div class="col-sm-6 mb-3">
                                <small class="text-muted d-block">Order Type</small>
                                <strong><?= ucfirst($order['order_type']) ?></strong>
                            </div>
                            <div class="col-sm-6 mb-3">
                                <small class="text-muted d-block">Payment Method</small>
                                <strong><?= ucfirst($order['payment_method']) ?></strong>
                            </div>
                            <?php if ($order['order_type'] === 'delivery'): ?>
                            <div class="col-12">
                                <small class="text-muted d-block">Delivery Address</small>
                                <strong><?= htmlspecialchars($order['delivery_address']) ?></strong>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Tracking Progress -->
                        <h6 class="mb-3"><i class="bi bi-geo-alt me-2"></i>Order Status</h6>
                        <div class="tracking-progress mb-4">
                            <div class="progress-track">
                                <div class="progress-fill" style="width: <?= ($currentStep / 6) * 100 ?>%"></div>
                            </div>
                            <div class="tracking-steps-horizontal">
                                <div class="step <?= $currentStep >= 1 ? 'completed' : '' ?> <?= $currentStep == 1 ? 'current' : '' ?>">
                                    <div class="step-icon"><i class="bi bi-clipboard-check"></i></div>
                                    <span>Placed</span>
                                </div>
                                <div class="step <?= $currentStep >= 2 ? 'completed' : '' ?> <?= $currentStep == 2 ? 'current' : '' ?>">
                                    <div class="step-icon"><i class="bi bi-check-circle"></i></div>
                                    <span>Confirmed</span>
                                </div>
                                <div class="step <?= $currentStep >= 3 ? 'completed' : '' ?> <?= $currentStep == 3 ? 'current' : '' ?>">
                                    <div class="step-icon"><i class="bi bi-fire"></i></div>
                                    <span>Preparing</span>
                                </div>
                                <div class="step <?= $currentStep >= 4 ? 'completed' : '' ?> <?= $currentStep == 4 ? 'current' : '' ?>">
                                    <div class="step-icon"><i class="bi bi-box-seam"></i></div>
                                    <span>Ready</span>
                                </div>
                                <div class="step <?= $currentStep >= 5 ? 'completed' : '' ?> <?= $currentStep == 5 ? 'current' : '' ?>">
                                    <div class="step-icon"><i class="bi bi-truck"></i></div>
                                    <span>On Way</span>
                                </div>
                                <div class="step <?= $currentStep >= 6 ? 'completed' : '' ?> <?= $currentStep == 6 ? 'current' : '' ?>">
                                    <div class="step-icon"><i class="bi bi-house-check"></i></div>
                                    <span>Delivered</span>
                                </div>
                            </div>
                        </div>
                        
                        <?php if ($order['order_status'] === 'cancelled'): ?>
                        <div class="alert alert-danger">
                            <i class="bi bi-x-circle me-2"></i>This order has been cancelled.
                        </div>
                        <?php elseif ($order['order_status'] === 'delivered'): ?>
                        <div class="alert alert-success">
                            <i class="bi bi-check-circle me-2"></i>Your order has been delivered successfully!
                        </div>
                        <?php else: ?>
                        <div class="alert alert-info">
                            <i class="bi bi-clock me-2"></i>
                            <strong>Estimated Delivery:</strong> 
                            <?= date('h:i A', strtotime($order['estimated_delivery_time'])) ?>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Order Items -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-bag me-2"></i>Order Items</h5>
                    </div>
                    <div class="card-body">
                        <?php foreach ($order['items'] as $item): ?>
                        <div class="d-flex justify-content-between align-items-center py-2 border-bottom">
                            <div>
                                <strong><?= htmlspecialchars($item['item_name']) ?></strong>
                                <span class="text-muted">× <?= $item['quantity'] ?></span>
                            </div>
                            <span><?= formatPrice($item['total_price']) ?></span>
                        </div>
                        <?php endforeach; ?>
                        
                        <div class="mt-3">
                            <div class="d-flex justify-content-between py-1">
                                <span>Subtotal</span>
                                <span><?= formatPrice($order['subtotal']) ?></span>
                            </div>
                            <div class="d-flex justify-content-between py-1">
                                <span>Delivery Fee</span>
                                <span><?= formatPrice($order['delivery_fee']) ?></span>
                            </div>
                            <div class="d-flex justify-content-between py-1">
                                <span>Tax</span>
                                <span><?= formatPrice($order['tax']) ?></span>
                            </div>
                            <?php if ($order['discount'] > 0): ?>
                            <div class="d-flex justify-content-between py-1 text-success">
                                <span>Discount</span>
                                <span>-<?= formatPrice($order['discount']) ?></span>
                            </div>
                            <?php endif; ?>
                            <hr>
                            <div class="d-flex justify-content-between py-1">
                                <strong>Total</strong>
                                <strong class="text-primary"><?= formatPrice($order['total_amount']) ?></strong>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Actions -->
                <div class="text-center">
                    <a href="track-order.php" class="btn btn-outline-primary me-2">
                        <i class="bi bi-search me-2"></i>Track Another Order
                    </a>
                    <a href="menu.php" class="btn btn-primary">
                        <i class="bi bi-grid me-2"></i>Order More
                    </a>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>
</section>

<style>
.tracking-progress {
    position: relative;
    padding: 20px 0;
}

.progress-track {
    position: absolute;
    top: 35px;
    left: 8%;
    right: 8%;
    height: 4px;
    background: #e0e0e0;
    border-radius: 2px;
    z-index: 1;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #28a745, #20c997);
    border-radius: 2px;
    transition: width 0.5s ease;
}

.tracking-steps-horizontal {
    display: flex;
    justify-content: space-between;
    position: relative;
    z-index: 2;
}

.tracking-steps-horizontal .step {
    text-align: center;
    flex: 1;
}

.tracking-steps-horizontal .step-icon {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: #e0e0e0;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 8px;
    color: #999;
    font-size: 18px;
    transition: all 0.3s ease;
}

.tracking-steps-horizontal .step.completed .step-icon {
    background: linear-gradient(135deg, #28a745, #20c997);
    color: white;
}

.tracking-steps-horizontal .step.current .step-icon {
    background: var(--primary-color);
    color: white;
    box-shadow: 0 0 0 4px rgba(220, 53, 69, 0.2);
    animation: pulse 2s infinite;
}

.tracking-steps-horizontal .step span {
    font-size: 12px;
    color: #666;
}

.tracking-steps-horizontal .step.completed span,
.tracking-steps-horizontal .step.current span {
    color: #333;
    font-weight: 500;
}

@keyframes pulse {
    0% { box-shadow: 0 0 0 0 rgba(220, 53, 69, 0.4); }
    70% { box-shadow: 0 0 0 10px rgba(220, 53, 69, 0); }
    100% { box-shadow: 0 0 0 0 rgba(220, 53, 69, 0); }
}

@media (max-width: 576px) {
    .tracking-steps-horizontal .step span {
        font-size: 10px;
    }
    .tracking-steps-horizontal .step-icon {
        width: 32px;
        height: 32px;
        font-size: 14px;
    }
}
</style>

<?php require_once 'includes/footer.php'; ?>

