<?php
$pageTitle = 'My Profile - OrderFood Portal';
require_once 'config.php';

// Redirect if not logged in
if (!isLoggedIn()) {
    header('Location: login.php?redirect=profile.php');
    exit;
}

$user = getUser();
$db = getDB();

// Handle profile update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $address = trim($_POST['address'] ?? '');
    $currentPassword = $_POST['current_password'] ?? '';
    $newPassword = $_POST['new_password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';
    
    $errors = [];
    
    if (empty($name)) {
        $errors[] = 'Name is required';
    }
    
    // Update profile
    if (empty($errors)) {
        try {
            // Update basic info
            $stmt = $db->prepare("UPDATE users SET name = ?, phone = ?, address = ? WHERE id = ?");
            $stmt->execute([$name, $phone, $address, $_SESSION['user_id']]);
            
            // Update password if provided
            if (!empty($newPassword)) {
                if (empty($currentPassword)) {
                    $errors[] = 'Current password is required to change password';
                } else {
                    // Verify current password
                    $userStmt = $db->prepare("SELECT password FROM users WHERE id = ?");
                    $userStmt->execute([$_SESSION['user_id']]);
                    $userData = $userStmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($userData && password_verify($currentPassword, $userData['password'])) {
                        if (strlen($newPassword) < 6) {
                            $errors[] = 'New password must be at least 6 characters';
                        } elseif ($newPassword !== $confirmPassword) {
                            $errors[] = 'New passwords do not match';
                        } else {
                            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
                            $passStmt = $db->prepare("UPDATE users SET password = ? WHERE id = ?");
                            $passStmt->execute([$hashedPassword, $_SESSION['user_id']]);
                        }
                    } else {
                        $errors[] = 'Current password is incorrect';
                    }
                }
            }
            
            if (empty($errors)) {
                // Update session
                $_SESSION['user_name'] = $name;
                $_SESSION['user_phone'] = $phone;
                $_SESSION['user_address'] = $address;
                
                setFlash('success', 'Profile updated successfully!');
                header('Location: profile.php');
                exit;
            }
        } catch(PDOException $e) {
            $errors[] = 'Error updating profile: ' . $e->getMessage();
        }
    }
    
    foreach ($errors as $error) {
        setFlash('danger', $error);
    }
}

// Get fresh user data
$stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$_SESSION['user_id']]);
$userData = $stmt->fetch(PDO::FETCH_ASSOC);

require_once 'includes/header.php';
?>

<!-- Page Header -->
<section class="page-header">
    <div class="container">
        <h1 data-aos="fade-up">My Profile</h1>
        <nav aria-label="breadcrumb" data-aos="fade-up" data-aos-delay="100">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="index.php">Home</a></li>
                <li class="breadcrumb-item active">My Profile</li>
            </ol>
        </nav>
    </div>
</section>

<!-- Profile Section -->
<section class="py-5">
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-lg-8">
                <div class="card shadow-sm">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0"><i class="bi bi-person-circle me-2"></i>Profile Information</h4>
                    </div>
                    <div class="card-body p-4">
                        <form method="POST" action="">
                            <!-- Personal Information -->
                            <h5 class="mb-4"><i class="bi bi-person me-2"></i>Personal Information</h5>
                            
                            <div class="form-floating mb-3">
                                <input type="text" class="form-control" id="name" name="name" 
                                       placeholder="Full Name" value="<?= htmlspecialchars($userData['name'] ?? '') ?>" required>
                                <label for="name">Full Name *</label>
                            </div>
                            
                            <div class="form-floating mb-3">
                                <input type="email" class="form-control" id="email" 
                                       value="<?= htmlspecialchars($userData['email'] ?? '') ?>" disabled>
                                <label for="email">Email Address</label>
                                <small class="text-muted">Email cannot be changed</small>
                            </div>
                            
                            <div class="form-floating mb-3">
                                <input type="tel" class="form-control" id="phone" name="phone" 
                                       placeholder="Phone Number" value="<?= htmlspecialchars($userData['phone'] ?? '') ?>" required>
                                <label for="phone">Phone Number *</label>
                            </div>
                            
                            <div class="form-floating mb-4">
                                <textarea class="form-control" id="address" name="address" 
                                          placeholder="Address" style="height: 100px"><?= htmlspecialchars($userData['address'] ?? '') ?></textarea>
                                <label for="address">Address</label>
                            </div>
                            
                            <hr class="my-4">
                            
                            <!-- Change Password -->
                            <h5 class="mb-4"><i class="bi bi-lock me-2"></i>Change Password</h5>
                            <p class="text-muted mb-3">Leave blank if you don't want to change password</p>
                            
                            <div class="form-floating mb-3">
                                <input type="password" class="form-control" id="current_password" name="current_password" 
                                       placeholder="Current Password">
                                <label for="current_password">Current Password</label>
                            </div>
                            
                            <div class="form-floating mb-3">
                                <input type="password" class="form-control" id="new_password" name="new_password" 
                                       placeholder="New Password" minlength="6">
                                <label for="new_password">New Password</label>
                            </div>
                            
                            <div class="form-floating mb-4">
                                <input type="password" class="form-control" id="confirm_password" name="confirm_password" 
                                       placeholder="Confirm New Password">
                                <label for="confirm_password">Confirm New Password</label>
                            </div>
                            
                            <div class="d-flex gap-3">
                                <button type="submit" class="btn btn-primary btn-lg">
                                    <i class="bi bi-check-circle me-2"></i>Update Profile
                                </button>
                                <a href="index.php" class="btn btn-outline-secondary btn-lg">
                                    <i class="bi bi-arrow-left me-2"></i>Cancel
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<?php require_once 'includes/footer.php'; ?>

