<?php
$pageTitle = 'My Orders - OrderFood Portal';
require_once 'config.php';

// Redirect if not logged in
if (!isLoggedIn()) {
    header('Location: login.php?redirect=orders.php');
    exit;
}

$db = getDB();
$userId = $_SESSION['user_id'];

// Check if location_id column exists in orders table
$columnExists = false;
try {
    $checkStmt = $db->query("SHOW COLUMNS FROM orders LIKE 'location_id'");
    $columnExists = $checkStmt->rowCount() > 0;
} catch(PDOException $e) {
    $columnExists = false;
}

// Get user orders - with or without location join based on column existence
if ($columnExists) {
    $stmt = $db->prepare("SELECT o.*, l.name as location_name 
                          FROM orders o 
                          LEFT JOIN locations l ON o.location_id = l.id
                          WHERE o.user_id = ? 
                          ORDER BY o.created_at DESC");
} else {
    $stmt = $db->prepare("SELECT o.*, NULL as location_name 
                          FROM orders o 
                          WHERE o.user_id = ? 
                          ORDER BY o.created_at DESC");
}
$stmt->execute([$userId]);
$orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get order items for each order
foreach ($orders as &$order) {
    $itemsStmt = $db->prepare("SELECT oi.*, m.name as item_name, m.image as item_image
                               FROM order_items oi 
                               LEFT JOIN menu_items m ON oi.menu_item_id = m.id 
                               WHERE oi.order_id = ?");
    $itemsStmt->execute([$order['id']]);
    $order['items'] = $itemsStmt->fetchAll(PDO::FETCH_ASSOC);
}

// Status badge colors
function getStatusBadge($status) {
    $badges = [
        'pending' => 'warning',
        'confirmed' => 'info',
        'preparing' => 'primary',
        'ready' => 'success',
        'out_for_delivery' => 'info',
        'delivered' => 'success',
        'cancelled' => 'danger'
    ];
    return $badges[$status] ?? 'secondary';
}

require_once 'includes/header.php';
?>

<!-- Page Header -->
<section class="page-header">
    <div class="container">
        <h1 data-aos="fade-up">My Orders</h1>
        <nav aria-label="breadcrumb" data-aos="fade-up" data-aos-delay="100">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="index.php">Home</a></li>
                <li class="breadcrumb-item active">My Orders</li>
            </ol>
        </nav>
    </div>
</section>

<!-- Orders Section -->
<section class="py-5">
    <div class="container">
        <?php if (count($orders) > 0): ?>
            <div class="row">
                <?php foreach ($orders as $order): ?>
                    <div class="col-12 mb-4">
                        <div class="card shadow-sm">
                            <div class="card-header bg-light">
                                <div class="row align-items-center">
                                    <div class="col-md-6">
                                        <h5 class="mb-0">
                                            <i class="bi bi-receipt me-2"></i>
                                            Order #<?= htmlspecialchars($order['order_number']) ?>
                                        </h5>
                                        <small class="text-muted">
                                            <i class="bi bi-calendar me-1"></i>
                                            <?= date('M d, Y h:i A', strtotime($order['created_at'])) ?>
                                        </small>
                                    </div>
                                    <div class="col-md-6 text-md-end">
                                        <span class="badge bg-<?= getStatusBadge($order['order_status']) ?> fs-6">
                                            <?= ucfirst(str_replace('_', ' ', $order['order_status'])) ?>
                                        </span>
                                        <?php if ($order['location_name']): ?>
                                            <br>
                                            <small class="text-muted">
                                                <i class="bi bi-geo-alt me-1"></i>
                                                <?= htmlspecialchars($order['location_name']) ?>
                                            </small>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <div class="card-body">
                                <!-- Order Items -->
                                <div class="table-responsive mb-3">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>Item</th>
                                                <th>Quantity</th>
                                                <th class="text-end">Price</th>
                                                <th class="text-end">Total</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($order['items'] as $item): ?>
                                                <tr>
                                                    <td>
                                                        <div class="d-flex align-items-center">
                                                            <?php if ($item['item_image']): ?>
                                                                <img src="uploads/<?= htmlspecialchars($item['item_image']) ?>" 
                                                                     alt="<?= htmlspecialchars($item['item_name']) ?>" 
                                                                     class="me-2" style="width: 50px; height: 50px; object-fit: cover; border-radius: 8px;">
                                                            <?php endif; ?>
                                                            <div>
                                                                <strong><?= htmlspecialchars($item['item_name']) ?></strong>
                                                                <?php if ($item['variant_name']): ?>
                                                                    <br><small class="text-muted"><?= htmlspecialchars($item['variant_name']) ?></small>
                                                                <?php endif; ?>
                                                            </div>
                                                        </div>
                                                    </td>
                                                    <td><?= $item['quantity'] ?></td>
                                                    <td class="text-end"><?= formatPrice($item['unit_price']) ?></td>
                                                    <td class="text-end"><strong><?= formatPrice($item['total_price']) ?></strong></td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                                
                                <!-- Order Summary -->
                                <div class="row">
                                    <div class="col-md-8">
                                        <div class="mb-2">
                                            <strong>Delivery Address:</strong><br>
                                            <?= nl2br(htmlspecialchars($order['delivery_address'])) ?>
                                        </div>
                                        <?php if ($order['order_notes']): ?>
                                            <div class="mb-2">
                                                <strong>Notes:</strong><br>
                                                <?= htmlspecialchars($order['order_notes']) ?>
                                            </div>
                                        <?php endif; ?>
                                        <div>
                                            <strong>Payment Method:</strong>
                                            <?= ucfirst($order['payment_method']) ?>
                                        </div>
                                    </div>
                                    <div class="col-md-4 text-md-end">
                                        <div class="mb-2">
                                            <small class="text-muted">Subtotal:</small><br>
                                            <strong><?= formatPrice($order['subtotal']) ?></strong>
                                        </div>
                                        <div class="mb-2">
                                            <small class="text-muted">Delivery Fee:</small><br>
                                            <strong><?= formatPrice($order['delivery_fee']) ?></strong>
                                        </div>
                                        <?php if ($order['tax'] > 0): ?>
                                            <div class="mb-2">
                                                <small class="text-muted">Tax:</small><br>
                                                <strong><?= formatPrice($order['tax']) ?></strong>
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($order['discount'] > 0): ?>
                                            <div class="mb-2 text-success">
                                                <small>Discount:</small><br>
                                                <strong>-<?= formatPrice($order['discount']) ?></strong>
                                            </div>
                                        <?php endif; ?>
                                        <hr>
                                        <div>
                                            <strong class="fs-5 text-primary">Total: <?= formatPrice($order['total_amount']) ?></strong>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Order Actions -->
                                <div class="mt-3 pt-3 border-top">
                                    <div class="d-flex gap-2">
                                        <a href="track-order.php?order=<?= urlencode($order['order_number']) ?>" 
                                           class="btn btn-outline-primary btn-sm">
                                            <i class="bi bi-truck me-1"></i>Track Order
                                        </a>
                                        <?php if ($order['order_status'] == 'pending' || $order['order_status'] == 'confirmed'): ?>
                                            <button class="btn btn-outline-danger btn-sm" 
                                                    onclick="cancelOrder('<?= $order['order_number'] ?>')">
                                                <i class="bi bi-x-circle me-1"></i>Cancel Order
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="empty-state text-center py-5">
                <i class="bi bi-bag-x" style="font-size: 80px; color: #ccc; margin-bottom: 20px;"></i>
                <h3>No Orders Yet</h3>
                <p class="text-muted">You haven't placed any orders yet. Start ordering delicious food now!</p>
                <a href="menu.php" class="btn btn-primary btn-lg mt-3">
                    <i class="bi bi-grid me-2"></i>Browse Menu
                </a>
            </div>
        <?php endif; ?>
    </div>
</section>

<script>
function cancelOrder(orderNumber) {
    if (confirm('Are you sure you want to cancel this order?')) {
        // You can implement order cancellation here
        alert('Order cancellation feature coming soon!');
    }
}
</script>

<?php require_once 'includes/footer.php'; ?>

