<?php
$pageTitle = 'Our Menu - OrderFood Portal';
require_once 'config.php';

$db = getDB();

// Get categories
$categoriesQuery = $db->query("SELECT * FROM categories WHERE is_active = 1 ORDER BY sort_order");
$categories = $categoriesQuery->fetchAll(PDO::FETCH_ASSOC);

// Get selected category
$selectedCategory = isset($_GET['category']) ? (int)$_GET['category'] : 0;

// Get menu items
$query = "SELECT m.*, c.name as category_name FROM menu_items m 
          JOIN categories c ON m.category_id = c.id 
          WHERE m.is_available = 1";

if ($selectedCategory > 0) {
    $query .= " AND m.category_id = :category_id";
}

// Search
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
if (!empty($search)) {
    $query .= " AND (m.name LIKE :search OR m.description LIKE :search)";
}

$query .= " ORDER BY m.is_featured DESC, m.sort_order ASC";

$stmt = $db->prepare($query);

if ($selectedCategory > 0) {
    $stmt->bindParam(':category_id', $selectedCategory);
}

if (!empty($search)) {
    $searchTerm = "%{$search}%";
    $stmt->bindParam(':search', $searchTerm);
}

$stmt->execute();
$menuItems = $stmt->fetchAll(PDO::FETCH_ASSOC);

require_once 'includes/header.php';
?>

<!-- Page Header -->
<section class="page-header">
    <div class="container">
        <h1 data-aos="fade-up">Our Menu</h1>
        <nav aria-label="breadcrumb" data-aos="fade-up" data-aos-delay="100">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="index.php">Home</a></li>
                <li class="breadcrumb-item active">Menu</li>
            </ol>
        </nav>
    </div>
</section>

<!-- Menu Section -->
<section class="menu-section">
    <div class="container">
        <!-- Search Bar -->
        <div class="row justify-content-center mb-5" data-aos="fade-up">
            <div class="col-lg-6">
                <form action="" method="GET" class="d-flex gap-2">
                    <?php if ($selectedCategory > 0): ?>
                        <input type="hidden" name="category" value="<?= $selectedCategory ?>">
                    <?php endif; ?>
                    <input type="text" name="search" class="form-control form-control-lg" placeholder="Search for food..." value="<?= htmlspecialchars($search) ?>">
                    <button type="submit" class="btn btn-primary btn-lg">
                        <i class="bi bi-search"></i>
                    </button>
                </form>
            </div>
        </div>
        
        <!-- Category Filters -->
        <div class="filter-tabs" data-aos="fade-up">
            <a href="menu.php" class="filter-tab <?= $selectedCategory == 0 ? 'active' : '' ?>">
                All Items
            </a>
            <?php foreach ($categories as $category): ?>
                <a href="menu.php?category=<?= $category['id'] ?>" class="filter-tab <?= $selectedCategory == $category['id'] ? 'active' : '' ?>">
                    <?= htmlspecialchars($category['name']) ?>
                </a>
            <?php endforeach; ?>
        </div>
        
        <!-- Menu Items -->
        <?php if (count($menuItems) > 0): ?>
            <div class="row g-4">
                <?php foreach ($menuItems as $index => $item): ?>
                    <div class="col-lg-3 col-md-6 menu-item" data-category="<?= $item['category_id'] ?>" data-aos="fade-up" data-aos-delay="<?= ($index % 4) * 50 ?>">
                        <div class="food-card">
                            <div class="image">
                                <img src="<?= $item['image'] ? 'uploads/' . htmlspecialchars($item['image']) : 'https://images.unsplash.com/photo-1565299624946-b28f40a0ae38?w=400&h=300&fit=crop' ?>" alt="<?= htmlspecialchars($item['name']) ?>">
                                <div class="badges">
                                    <?php if ($item['is_featured']): ?>
                                        <span class="badge featured">⭐ Featured</span>
                                    <?php endif; ?>
                                    <?php if ($item['is_vegetarian']): ?>
                                        <span class="badge vegetarian">🥬 Veg</span>
                                    <?php endif; ?>
                                    <?php if ($item['is_spicy']): ?>
                                        <span class="badge spicy">🌶️ Spicy</span>
                                    <?php endif; ?>
                                    <?php if ($item['discount_price']): ?>
                                        <span class="badge discount"><?= round((($item['price'] - $item['discount_price']) / $item['price']) * 100) ?>% OFF</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="content">
                                <span class="category"><?= htmlspecialchars($item['category_name']) ?></span>
                                <h4><?= htmlspecialchars($item['name']) ?></h4>
                                <p class="description"><?= htmlspecialchars($item['description']) ?></p>
                                <div class="footer">
                                    <div class="price">
                                        <?php if ($item['discount_price']): ?>
                                            <span class="current"><?= formatPrice($item['discount_price']) ?></span>
                                            <span class="original"><?= formatPrice($item['price']) ?></span>
                                        <?php else: ?>
                                            <span class="current"><?= formatPrice($item['price']) ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <button class="add-btn add-to-cart" data-item-id="<?= $item['id'] ?>">
                                        <i class="bi bi-plus"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="empty-state">
                <i class="bi bi-search"></i>
                <h3>No items found</h3>
                <p>Try a different search term or browse all categories</p>
                <a href="menu.php" class="btn btn-primary">View All Items</a>
            </div>
        <?php endif; ?>
    </div>
</section>

<?php require_once 'includes/footer.php'; ?>

