<?php
/**
 * Cart Handler - AJAX operations for cart
 */
require_once 'config.php';

header('Content-Type: application/json');

$response = ['success' => false];

// Handle GET requests (clear cart)
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $action = $_GET['action'] ?? '';
    
    if ($action === 'clear') {
        $_SESSION['cart'] = [];
        $_SESSION['coupon_code'] = null;
        $_SESSION['discount'] = 0;
        setFlash('success', 'Cart cleared successfully');
        header('Location: cart.php');
        exit;
    }
}

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'add':
            $itemId = (int)$_POST['item_id'];
            $variantId = !empty($_POST['variant_id']) ? (int)$_POST['variant_id'] : null;
            $quantity = (int)($_POST['quantity'] ?? 1);
            
            if ($itemId > 0 && $quantity > 0) {
                // Verify item exists
                $db = getDB();
                $stmt = $db->prepare("SELECT id, name FROM menu_items WHERE id = ? AND is_available = 1");
                $stmt->execute([$itemId]);
                $item = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($item) {
                    // Create unique key for cart item
                    $cartKey = $itemId . '_' . ($variantId ?? '0');
                    
                    // Initialize cart if not exists
                    if (!isset($_SESSION['cart'])) {
                        $_SESSION['cart'] = [];
                    }
                    
                    // Add or update item
                    if (isset($_SESSION['cart'][$cartKey])) {
                        $_SESSION['cart'][$cartKey]['quantity'] += $quantity;
                    } else {
                        $_SESSION['cart'][$cartKey] = [
                            'item_id' => $itemId,
                            'variant_id' => $variantId,
                            'quantity' => $quantity
                        ];
                    }
                    
                    $response = [
                        'success' => true,
                        'message' => 'Item added to cart',
                        'cart_count' => getCartCount()
                    ];
                } else {
                    $response['message'] = 'Item not found or unavailable';
                }
            } else {
                $response['message'] = 'Invalid item or quantity';
            }
            break;
            
        case 'increase':
            $key = $_POST['key'] ?? '';
            if (isset($_SESSION['cart'][$key])) {
                $_SESSION['cart'][$key]['quantity']++;
                $response = ['success' => true, 'cart_count' => getCartCount()];
            }
            break;
            
        case 'decrease':
            $key = $_POST['key'] ?? '';
            if (isset($_SESSION['cart'][$key])) {
                $_SESSION['cart'][$key]['quantity']--;
                if ($_SESSION['cart'][$key]['quantity'] <= 0) {
                    unset($_SESSION['cart'][$key]);
                }
                $response = ['success' => true, 'cart_count' => getCartCount()];
            }
            break;
            
        case 'remove':
            $key = $_POST['key'] ?? '';
            if (isset($_SESSION['cart'][$key])) {
                unset($_SESSION['cart'][$key]);
                $response = ['success' => true, 'cart_count' => getCartCount()];
            }
            break;
            
        case 'apply_coupon':
            $code = strtoupper(trim($_POST['code'] ?? ''));
            
            if (!empty($code)) {
                $db = getDB();
                $stmt = $db->prepare("SELECT * FROM coupons WHERE code = ? AND is_active = 1 AND (start_date IS NULL OR start_date <= CURDATE()) AND (end_date IS NULL OR end_date >= CURDATE())");
                $stmt->execute([$code]);
                $coupon = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($coupon) {
                    // Check usage limit
                    if ($coupon['usage_limit'] && $coupon['used_count'] >= $coupon['usage_limit']) {
                        $response['message'] = 'This coupon has been fully redeemed';
                    } else {
                        // Calculate discount
                        $subtotal = getCartTotal();
                        
                        if ($subtotal < $coupon['min_order_amount']) {
                            $response['message'] = 'Minimum order amount is ' . formatPrice($coupon['min_order_amount']);
                        } else {
                            if ($coupon['discount_type'] === 'percentage') {
                                $discount = ($subtotal * $coupon['discount_value']) / 100;
                                if ($coupon['max_discount'] && $discount > $coupon['max_discount']) {
                                    $discount = $coupon['max_discount'];
                                }
                            } else {
                                $discount = $coupon['discount_value'];
                            }
                            
                            $_SESSION['coupon_code'] = $code;
                            $_SESSION['coupon_id'] = $coupon['id'];
                            $_SESSION['discount'] = $discount;
                            
                            $response = [
                                'success' => true,
                                'message' => 'Coupon applied! You saved ' . formatPrice($discount),
                                'discount' => $discount
                            ];
                        }
                    }
                } else {
                    $response['message'] = 'Invalid or expired coupon code';
                }
            } else {
                $response['message'] = 'Please enter a coupon code';
            }
            break;
            
        default:
            $response['message'] = 'Invalid action';
    }
}

echo json_encode($response);

