/**
 * OrderFood Portal - Main JavaScript
 */

document.addEventListener('DOMContentLoaded', function() {
    // Initialize components
    initNavbar();
    initCart();
    initForms();
    initAnimations();
    initSmoothLinks();
});

/**
 * Smooth link navigation - prevent page jump
 */
function initSmoothLinks() {
    // Prevent page jump on auth page links
    document.querySelectorAll('.smooth-link').forEach(link => {
        link.addEventListener('click', function(e) {
            // Allow normal navigation but prevent scroll jump
            window.scrollTo(0, 0);
        });
    });
    
    // Smooth scroll to top when navigating between auth pages
    if (window.location.pathname.includes('login.php') || 
        window.location.pathname.includes('register.php')) {
        window.scrollTo({ top: 0, behavior: 'smooth' });
    }
}

/**
 * Navbar scroll effect
 */
function initNavbar() {
    const navbar = document.querySelector('.navbar');
    
    if (navbar) {
        window.addEventListener('scroll', function() {
            if (window.scrollY > 50) {
                navbar.classList.add('scrolled');
            } else {
                navbar.classList.remove('scrolled');
            }
        });
        
        // Trigger on load if already scrolled
        if (window.scrollY > 50) {
            navbar.classList.add('scrolled');
        }
    }
}

/**
 * Cart functionality
 */
function initCart() {
    // Add to cart buttons
    document.querySelectorAll('.add-to-cart').forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            const itemId = this.dataset.itemId;
            const variantId = this.dataset.variantId || null;
            addToCart(itemId, variantId, 1);
        });
    });
    
    // Quantity controls
    document.querySelectorAll('.qty-btn').forEach(button => {
        button.addEventListener('click', function() {
            const action = this.dataset.action;
            const itemKey = this.dataset.key;
            updateCartQuantity(itemKey, action);
        });
    });
    
    // Remove from cart
    document.querySelectorAll('.remove-from-cart').forEach(button => {
        button.addEventListener('click', function() {
            const itemKey = this.dataset.key;
            removeFromCart(itemKey);
        });
    });
}

/**
 * Add item to cart
 */
function addToCart(itemId, variantId, quantity) {
    const formData = new FormData();
    formData.append('action', 'add');
    formData.append('item_id', itemId);
    formData.append('variant_id', variantId);
    formData.append('quantity', quantity);
    
    fetch('cart-handler.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            updateCartBadge(data.cart_count);
            showNotification('success', 'Item added to cart!');
            
            // Animate cart icon
            const cartIcon = document.querySelector('.cart-icon');
            if (cartIcon) {
                cartIcon.classList.add('bounce');
                setTimeout(() => cartIcon.classList.remove('bounce'), 500);
            }
        } else {
            showNotification('error', data.message || 'Failed to add item');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('error', 'Something went wrong');
    });
}

/**
 * Update cart quantity
 */
function updateCartQuantity(itemKey, action) {
    const formData = new FormData();
    formData.append('action', action);
    formData.append('key', itemKey);
    
    fetch('cart-handler.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        }
    })
    .catch(error => {
        console.error('Error:', error);
    });
}

/**
 * Remove item from cart
 */
function removeFromCart(itemKey) {
    if (confirm('Remove this item from cart?')) {
        const formData = new FormData();
        formData.append('action', 'remove');
        formData.append('key', itemKey);
        
        fetch('cart-handler.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            }
        })
        .catch(error => {
            console.error('Error:', error);
        });
    }
}

/**
 * Update cart badge count
 */
function updateCartBadge(count) {
    const badge = document.getElementById('cartCount');
    if (badge) {
        badge.textContent = count;
        badge.classList.add('bounce');
        setTimeout(() => badge.classList.remove('bounce'), 300);
    }
}

/**
 * Show notification
 */
function showNotification(type, message) {
    // Remove existing notifications
    document.querySelectorAll('.notification').forEach(n => n.remove());
    
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <i class="bi bi-${type === 'success' ? 'check-circle' : 'exclamation-circle'}"></i>
        <span>${message}</span>
    `;
    
    document.body.appendChild(notification);
    
    // Animate in
    setTimeout(() => notification.classList.add('show'), 10);
    
    // Remove after 3 seconds
    setTimeout(() => {
        notification.classList.remove('show');
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

// Add notification styles dynamically
const notificationStyles = document.createElement('style');
notificationStyles.textContent = `
    .notification {
        position: fixed;
        top: 100px;
        right: 20px;
        background: #fff;
        padding: 15px 25px;
        border-radius: 10px;
        box-shadow: 0 10px 40px rgba(0,0,0,0.15);
        display: flex;
        align-items: center;
        gap: 12px;
        transform: translateX(120%);
        transition: transform 0.3s ease;
        z-index: 9999;
    }
    .notification.show {
        transform: translateX(0);
    }
    .notification-success {
        border-left: 4px solid #28a745;
    }
    .notification-success i {
        color: #28a745;
        font-size: 24px;
    }
    .notification-error {
        border-left: 4px solid #dc3545;
    }
    .notification-error i {
        color: #dc3545;
        font-size: 24px;
    }
    .cart-icon.bounce, .cart-badge.bounce {
        animation: bounce 0.5s ease;
    }
    @keyframes bounce {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.3); }
    }
`;
document.head.appendChild(notificationStyles);

/**
 * Form handling
 */
function initForms() {
    // Coupon form
    const couponForm = document.getElementById('couponForm');
    if (couponForm) {
        couponForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const code = this.querySelector('input[name="coupon"]').value;
            applyCoupon(code);
        });
    }
    
    // Contact form
    const contactForm = document.getElementById('contactForm');
    if (contactForm) {
        contactForm.addEventListener('submit', function(e) {
            e.preventDefault();
            submitContactForm(this);
        });
    }
}

/**
 * Apply coupon code
 */
function applyCoupon(code) {
    const formData = new FormData();
    formData.append('action', 'apply_coupon');
    formData.append('code', code);
    
    fetch('cart-handler.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('success', data.message);
            location.reload();
        } else {
            showNotification('error', data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
    });
}

/**
 * Submit contact form
 */
function submitContactForm(form) {
    const formData = new FormData(form);
    
    fetch('contact-handler.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('success', 'Message sent successfully!');
            form.reset();
        } else {
            showNotification('error', data.message || 'Failed to send message');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('error', 'Something went wrong');
    });
}

/**
 * Initialize animations
 */
function initAnimations() {
    // Intersection Observer for scroll animations
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate-in');
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);
    
    document.querySelectorAll('.animate-on-scroll').forEach(el => {
        observer.observe(el);
    });
}

/**
 * Menu filter functionality
 */
function filterMenu(categoryId) {
    const items = document.querySelectorAll('.menu-item');
    const tabs = document.querySelectorAll('.filter-tab');
    
    // Update active tab
    tabs.forEach(tab => {
        if (tab.dataset.category === categoryId || (categoryId === 'all' && !tab.dataset.category)) {
            tab.classList.add('active');
        } else {
            tab.classList.remove('active');
        }
    });
    
    // Filter items
    items.forEach(item => {
        if (categoryId === 'all' || item.dataset.category === categoryId) {
            item.style.display = 'block';
            item.classList.add('fade-in');
        } else {
            item.style.display = 'none';
        }
    });
}

/**
 * Order tracking
 */
function trackOrder(orderNumber) {
    const formData = new FormData();
    formData.append('order_number', orderNumber);
    
    fetch('track-order.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            updateTrackingUI(data.order);
        } else {
            showNotification('error', data.message || 'Order not found');
        }
    })
    .catch(error => {
        console.error('Error:', error);
    });
}

/**
 * Update tracking UI
 */
function updateTrackingUI(order) {
    const statuses = ['pending', 'confirmed', 'preparing', 'ready', 'out_for_delivery', 'delivered'];
    const currentIndex = statuses.indexOf(order.order_status);
    
    document.querySelectorAll('.tracking-step').forEach((step, index) => {
        step.classList.remove('completed', 'current');
        
        if (index < currentIndex) {
            step.classList.add('completed');
        } else if (index === currentIndex) {
            step.classList.add('current');
        }
    });
}

/**
 * Smooth scroll to element
 */
function scrollToElement(elementId) {
    const element = document.getElementById(elementId);
    if (element) {
        element.scrollIntoView({
            behavior: 'smooth',
            block: 'start'
        });
    }
}

/**
 * Format price with currency
 */
function formatPrice(amount) {
    return '$' + Number(amount).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2});
}

