<?php
/**
 * Get Orders
 * GET /api/orders/read.php
 * GET /api/orders/read.php?id=1
 * GET /api/orders/read.php?order_number=ORD20251202001
 * GET /api/orders/read.php?status=pending
 */

include_once '../config/cors.php';
include_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

try {
    $query = "SELECT o.* FROM orders o WHERE 1=1";
    
    // Single order by ID
    if (isset($_GET['id'])) {
        $query .= " AND o.id = :id";
    }
    
    // Single order by order number
    if (isset($_GET['order_number'])) {
        $query .= " AND o.order_number = :order_number";
    }
    
    // Filter by status
    if (isset($_GET['status'])) {
        $query .= " AND o.order_status = :status";
    }
    
    // Filter by date range
    if (isset($_GET['date_from'])) {
        $query .= " AND DATE(o.created_at) >= :date_from";
    }
    if (isset($_GET['date_to'])) {
        $query .= " AND DATE(o.created_at) <= :date_to";
    }
    
    $query .= " ORDER BY o.created_at DESC";
    
    // Pagination
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
    $offset = ($page - 1) * $limit;
    
    if (!isset($_GET['id']) && !isset($_GET['order_number'])) {
        $query .= " LIMIT :limit OFFSET :offset";
    }
    
    $stmt = $db->prepare($query);
    
    if (isset($_GET['id'])) {
        $stmt->bindParam(':id', $_GET['id']);
    }
    if (isset($_GET['order_number'])) {
        $stmt->bindParam(':order_number', $_GET['order_number']);
    }
    if (isset($_GET['status'])) {
        $stmt->bindParam(':status', $_GET['status']);
    }
    if (isset($_GET['date_from'])) {
        $stmt->bindParam(':date_from', $_GET['date_from']);
    }
    if (isset($_GET['date_to'])) {
        $stmt->bindParam(':date_to', $_GET['date_to']);
    }
    if (!isset($_GET['id']) && !isset($_GET['order_number'])) {
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
    }
    
    $stmt->execute();
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Fetch items for each order
    foreach ($orders as &$order) {
        $itemsQuery = "SELECT oi.*, m.name as item_name, v.name as variant_name 
                       FROM order_items oi 
                       LEFT JOIN menu_items m ON oi.menu_item_id = m.id
                       LEFT JOIN item_variants v ON oi.variant_id = v.id
                       WHERE oi.order_id = :order_id";
        $itemsStmt = $db->prepare($itemsQuery);
        $itemsStmt->bindParam(':order_id', $order['id']);
        $itemsStmt->execute();
        $order['items'] = $itemsStmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // If single order requested
    if (isset($_GET['id']) || isset($_GET['order_number'])) {
        if (count($orders) > 0) {
            http_response_code(200);
            echo json_encode([
                "success" => true,
                "data" => $orders[0]
            ]);
        } else {
            http_response_code(404);
            echo json_encode([
                "success" => false,
                "message" => "Order not found"
            ]);
        }
    } else {
        // Get total count for pagination
        $countQuery = "SELECT COUNT(*) as total FROM orders WHERE 1=1";
        if (isset($_GET['status'])) {
            $countQuery .= " AND order_status = :status";
        }
        $countStmt = $db->prepare($countQuery);
        if (isset($_GET['status'])) {
            $countStmt->bindParam(':status', $_GET['status']);
        }
        $countStmt->execute();
        $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        http_response_code(200);
        echo json_encode([
            "success" => true,
            "data" => $orders,
            "pagination" => [
                "current_page" => $page,
                "per_page" => $limit,
                "total" => (int)$total,
                "total_pages" => ceil($total / $limit)
            ]
        ]);
    }
} catch(PDOException $e) {
    http_response_code(500);
    echo json_encode([
        "success" => false,
        "message" => "Error fetching orders: " . $e->getMessage()
    ]);
}
