<?php
$pageTitle = 'Checkout - OrderFood Portal';
require_once 'config.php';

$db = getDB();
$cart = getCart();

// Check if location_id column exists in orders table
$locationColumnExists = false;
try {
    $checkStmt = $db->query("SHOW COLUMNS FROM orders LIKE 'location_id'");
    $locationColumnExists = $checkStmt->rowCount() > 0;
} catch(PDOException $e) {
    $locationColumnExists = false;
}

// Get all active locations only if column exists
$locations = [];
if ($locationColumnExists) {
    try {
        $locationsStmt = $db->query("SELECT * FROM locations WHERE is_active = 1 ORDER BY name");
        $locations = $locationsStmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        $locations = [];
    }
}

// Redirect if cart is empty
if (empty($cart)) {
    setFlash('warning', 'Your cart is empty. Please add items before checkout.');
    header('Location: menu.php');
    exit;
}

// Calculate totals
$subtotal = 0;
$cartItems = [];

foreach ($cart as $key => $item) {
    $stmt = $db->prepare("SELECT m.*, c.name as category_name FROM menu_items m JOIN categories c ON m.category_id = c.id WHERE m.id = ?");
    $stmt->execute([$item['item_id']]);
    $menuItem = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($menuItem) {
        $price = $menuItem['discount_price'] ?? $menuItem['price'];
        $totalPrice = $price * $item['quantity'];
        $subtotal += $totalPrice;
        
        $cartItems[$key] = [
            'item' => $menuItem,
            'quantity' => $item['quantity'],
            'unit_price' => $price,
            'total_price' => $totalPrice
        ];
    }
}

$deliveryFee = 100;
$tax = $subtotal * 0.05;
$discount = $_SESSION['discount'] ?? 0;
$total = $subtotal + $deliveryFee + $tax - $discount;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $customerName = trim($_POST['name'] ?? '');
    $customerEmail = trim($_POST['email'] ?? '');
    $customerPhone = trim($_POST['phone'] ?? '');
    $streetAddress = trim($_POST['street_address'] ?? '');
    $city = trim($_POST['city'] ?? '');
    $state = trim($_POST['state'] ?? '');
    $country = trim($_POST['country'] ?? '');
    $zipCode = trim($_POST['zip_code'] ?? '');
    $paymentMethod = $_POST['payment_method'] ?? 'cash';
    $orderType = $_POST['order_type'] ?? 'delivery';
    $locationId = !empty($_POST['location_id']) ? (int)$_POST['location_id'] : null;
    $specialInstructions = trim($_POST['instructions'] ?? '');
    
    // Combine address fields
    $addressParts = [];
    if (!empty($streetAddress)) $addressParts[] = $streetAddress;
    if (!empty($city)) $addressParts[] = $city;
    if (!empty($state)) $addressParts[] = $state;
    if (!empty($zipCode)) $addressParts[] = $zipCode;
    if (!empty($country)) $addressParts[] = $country;
    $deliveryAddress = implode(', ', $addressParts);
    
    $errors = [];
    
    if (empty($customerName)) $errors[] = 'Name is required';
    if (empty($customerPhone)) $errors[] = 'Phone is required';
    
    // Check if location_id column exists before requiring it
    $columnExists = false;
    try {
        $checkStmt = $db->query("SHOW COLUMNS FROM orders LIKE 'location_id'");
        $columnExists = $checkStmt->rowCount() > 0;
    } catch(PDOException $e) {
        $columnExists = false;
    }
    
    // Only require location if column exists
    if ($columnExists && empty($locationId)) {
        $errors[] = 'Please select a branch location';
    }
    
    if ($orderType === 'delivery') {
        if (empty($streetAddress)) $errors[] = 'Street address is required';
        if (empty($city)) $errors[] = 'City is required';
        if (empty($state)) $errors[] = 'State/Province is required';
        if (empty($country)) $errors[] = 'Country is required';
        if (empty($zipCode)) $errors[] = 'Zip/Postal code is required';
    }
    
    if (empty($errors)) {
        try {
            $db->beginTransaction();
            
            // Generate order number
            $orderNumber = 'ORD' . date('Ymd') . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
            
            // Estimated delivery time (45 minutes from now)
            $estimatedDelivery = date('Y-m-d H:i:s', strtotime('+45 minutes'));
            
            // Check if location_id column exists
            $columnExists = false;
            try {
                $checkStmt = $db->query("SHOW COLUMNS FROM orders LIKE 'location_id'");
                $columnExists = $checkStmt->rowCount() > 0;
            } catch(PDOException $e) {
                $columnExists = false;
            }
            
            $userId = isLoggedIn() ? $_SESSION['user_id'] : null;
            
            // Create order - with or without location_id based on column existence
            if ($columnExists) {
                $stmt = $db->prepare("INSERT INTO orders 
                    (order_number, user_id, customer_name, customer_email, customer_phone, delivery_address, 
                     subtotal, delivery_fee, tax, discount, total_amount, payment_method, order_type, location_id,
                     special_instructions, estimated_delivery_time) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                
                $stmt->execute([
                    $orderNumber, $userId, $customerName, $customerEmail, $customerPhone, $deliveryAddress,
                    $subtotal, $deliveryFee, $tax, $discount, $total, $paymentMethod, $orderType, $locationId,
                    $specialInstructions, $estimatedDelivery
                ]);
            } else {
                $stmt = $db->prepare("INSERT INTO orders 
                    (order_number, user_id, customer_name, customer_email, customer_phone, delivery_address, 
                     subtotal, delivery_fee, tax, discount, total_amount, payment_method, order_type,
                     special_instructions, estimated_delivery_time) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                
                $stmt->execute([
                    $orderNumber, $userId, $customerName, $customerEmail, $customerPhone, $deliveryAddress,
                    $subtotal, $deliveryFee, $tax, $discount, $total, $paymentMethod, $orderType,
                    $specialInstructions, $estimatedDelivery
                ]);
            }
            
            $orderId = $db->lastInsertId();
            
            // Insert order items
            $itemStmt = $db->prepare("INSERT INTO order_items (order_id, menu_item_id, variant_id, quantity, unit_price, total_price) VALUES (?, ?, ?, ?, ?, ?)");
            
            foreach ($cart as $cartItem) {
                $stmt2 = $db->prepare("SELECT * FROM menu_items WHERE id = ?");
                $stmt2->execute([$cartItem['item_id']]);
                $menuItem = $stmt2->fetch(PDO::FETCH_ASSOC);
                
                if ($menuItem) {
                    $price = $menuItem['discount_price'] ?? $menuItem['price'];
                    $totalPrice = $price * $cartItem['quantity'];
                    
                    // Handle variant_id - convert 0 or empty to null
                    $variantId = isset($cartItem['variant_id']) && !empty($cartItem['variant_id']) && $cartItem['variant_id'] != '0' 
                        ? $cartItem['variant_id'] 
                        : null;
                    
                    $itemStmt->execute([
                        $orderId,
                        $cartItem['item_id'],
                        $variantId,
                        $cartItem['quantity'],
                        $price,
                        $totalPrice
                    ]);
                }
            }
            
            // Update coupon usage if applied
            if (isset($_SESSION['coupon_id'])) {
                $db->prepare("UPDATE coupons SET used_count = used_count + 1 WHERE id = ?")->execute([$_SESSION['coupon_id']]);
            }
            
            $db->commit();
            
            // Clear cart and coupon
            $_SESSION['cart'] = [];
            unset($_SESSION['coupon_code'], $_SESSION['coupon_id'], $_SESSION['discount']);
            
            // Store order info for confirmation page
            $_SESSION['last_order'] = [
                'order_number' => $orderNumber,
                'total' => $total,
                'estimated_delivery' => $estimatedDelivery
            ];
            
            header('Location: order-success.php');
            exit;
            
        } catch (Exception $e) {
            $db->rollBack();
            $errors[] = 'Failed to place order: ' . $e->getMessage();
        }
    }
    
    if (!empty($errors)) {
        foreach ($errors as $error) {
            setFlash('danger', $error);
        }
    }
}

$user = getUser();

require_once 'includes/header.php';
?>

<!-- Page Header -->
<section class="page-header">
    <div class="container">
        <h1 data-aos="fade-up">Checkout</h1>
        <nav aria-label="breadcrumb" data-aos="fade-up" data-aos-delay="100">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="index.php">Home</a></li>
                <li class="breadcrumb-item"><a href="cart.php">Cart</a></li>
                <li class="breadcrumb-item active">Checkout</li>
            </ol>
        </nav>
    </div>
</section>

<!-- Checkout Section -->
<section class="checkout-section">
    <div class="container">
        <form method="POST" action="">
            <div class="row">
                <!-- Checkout Form -->
                <div class="col-lg-8" data-aos="fade-right">
                    <div class="checkout-form">
                        <!-- Contact Information -->
                        <h4><i class="bi bi-person me-2"></i>Contact Information</h4>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-floating">
                                    <input type="text" class="form-control" id="name" name="name" placeholder="Full Name" value="<?= htmlspecialchars($user['name'] ?? '') ?>" required>
                                    <label for="name">Full Name *</label>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-floating">
                                    <input type="tel" class="form-control" id="phone" name="phone" placeholder="Phone Number" value="<?= htmlspecialchars($user['phone'] ?? '') ?>" required>
                                    <label for="phone">Phone Number *</label>
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-floating">
                            <input type="email" class="form-control" id="email" name="email" placeholder="Email Address" value="<?= htmlspecialchars($user['email'] ?? '') ?>">
                            <label for="email">Email Address (Optional)</label>
                        </div>
                        
                        <!-- Branch Location (only show if column exists) -->
                        <?php if ($locationColumnExists && !empty($locations)): ?>
                            <h4 class="mt-4"><i class="bi bi-geo-alt me-2"></i>Select Branch Location</h4>
                            
                            <div class="form-floating mb-3">
                                <select class="form-select" id="location_id" name="location_id" required>
                                    <option value="">Select a branch location</option>
                                    <?php foreach ($locations as $location): ?>
                                        <option value="<?= $location['id'] ?>">
                                            <?= htmlspecialchars($location['name']) ?> - <?= htmlspecialchars($location['city']) ?>, <?= htmlspecialchars($location['state']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <label for="location_id">Branch Location *</label>
                            </div>
                        <?php endif; ?>
                        
                        <!-- Order Type -->
                        <h4 class="mt-4"><i class="bi bi-truck me-2"></i>Order Type</h4>
                        
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <div class="form-check p-3 border rounded">
                                    <input class="form-check-input" type="radio" name="order_type" id="delivery" value="delivery" checked>
                                    <label class="form-check-label w-100" for="delivery">
                                        <strong><i class="bi bi-truck me-2"></i>Delivery</strong>
                                        <small class="text-muted d-block">Delivered to your address</small>
                                    </label>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-check p-3 border rounded">
                                    <input class="form-check-input" type="radio" name="order_type" id="pickup" value="pickup">
                                    <label class="form-check-label w-100" for="pickup">
                                        <strong><i class="bi bi-shop me-2"></i>Pickup</strong>
                                        <small class="text-muted d-block">Pick up from selected branch</small>
                                    </label>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Delivery Address -->
                        <div id="addressSection">
                            <h4 class="mt-4"><i class="bi bi-geo-alt me-2"></i>Delivery Address</h4>
                            
                            <div class="form-floating mb-3">
                                <input type="text" class="form-control" id="street_address" name="street_address" placeholder="Street Address" value="<?= htmlspecialchars($user['address'] ?? '') ?>" required>
                                <label for="street_address">Street Address *</label>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <div class="form-floating">
                                        <input type="text" class="form-control" id="city" name="city" placeholder="City" required>
                                        <label for="city">City *</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-floating">
                                        <input type="text" class="form-control" id="state" name="state" placeholder="State/Province" required>
                                        <label for="state">State/Province *</label>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <div class="form-floating">
                                        <input type="text" class="form-control" id="zip_code" name="zip_code" placeholder="Zip/Postal Code" required>
                                        <label for="zip_code">Zip/Postal Code *</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-floating">
                                        <select class="form-select" id="country" name="country" required>
                                            <option value="">Select Country</option>
                                            <option value="Pakistan" selected>Pakistan</option>
                                            <option value="India">India</option>
                                            <option value="Bangladesh">Bangladesh</option>
                                            <option value="Sri Lanka">Sri Lanka</option>
                                            <option value="Afghanistan">Afghanistan</option>
                                            <option value="United States">United States</option>
                                            <option value="United Kingdom">United Kingdom</option>
                                            <option value="Canada">Canada</option>
                                            <option value="Australia">Australia</option>
                                            <option value="UAE">UAE</option>
                                            <option value="Saudi Arabia">Saudi Arabia</option>
                                            <option value="Other">Other</option>
                                        </select>
                                        <label for="country">Country *</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Payment Method -->
                        <h4 class="mt-4"><i class="bi bi-credit-card me-2"></i>Payment Method</h4>
                        
                        <div class="payment-methods mb-3">
                            <label class="payment-method active">
                                <input type="radio" name="payment_method" value="cash" checked hidden>
                                <i class="bi bi-cash"></i>
                                <span>Cash on Delivery</span>
                            </label>
                            <label class="payment-method">
                                <input type="radio" name="payment_method" value="card" hidden>
                                <i class="bi bi-credit-card"></i>
                                <span>Credit Card</span>
                            </label>
                        </div>
                        
                        <!-- Special Instructions -->
                        <h4 class="mt-4"><i class="bi bi-chat-text me-2"></i>Special Instructions</h4>
                        
                        <div class="form-floating">
                            <textarea class="form-control" id="instructions" name="instructions" placeholder="Special Instructions" style="height: 100px"></textarea>
                            <label for="instructions">Any special requests? (Optional)</label>
                        </div>
                    </div>
                </div>
                
                <!-- Order Summary -->
                <div class="col-lg-4" data-aos="fade-left">
                    <div class="cart-summary">
                        <h4>Order Summary</h4>
                        
                        <!-- Items -->
                        <div class="mb-4">
                            <?php foreach ($cartItems as $cartItem): ?>
                                <div class="d-flex justify-content-between mb-2">
                                    <span>
                                        <?= htmlspecialchars($cartItem['item']['name']) ?>
                                        <small class="text-muted">x<?= $cartItem['quantity'] ?></small>
                                    </span>
                                    <span><?= formatPrice($cartItem['total_price']) ?></span>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        
                        <hr>
                        
                        <div class="row">
                            <span>Subtotal</span>
                            <span><?= formatPrice($subtotal) ?></span>
                        </div>
                        <div class="row">
                            <span>Delivery Fee</span>
                            <span><?= formatPrice($deliveryFee) ?></span>
                        </div>
                        <div class="row">
                            <span>Tax (5%)</span>
                            <span><?= formatPrice($tax) ?></span>
                        </div>
                        <?php if ($discount > 0): ?>
                        <div class="row text-success">
                            <span>Discount</span>
                            <span>-<?= formatPrice($discount) ?></span>
                        </div>
                        <?php endif; ?>
                        <div class="row total">
                            <span>Total</span>
                            <span><?= formatPrice($total) ?></span>
                        </div>
                        
                        <button type="submit" class="btn btn-primary btn-lg w-100 mt-4">
                            <i class="bi bi-check-circle me-2"></i>Place Order
                        </button>
                        
                        <a href="cart.php" class="btn btn-outline-secondary w-100 mt-2">
                            <i class="bi bi-arrow-left me-2"></i>Back to Cart
                        </a>
                        
                        <div class="text-center mt-3">
                            <small class="text-muted">
                                <i class="bi bi-shield-check me-1"></i>
                                Your personal data is secure
                            </small>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>
</section>

<script>
// Toggle address section based on order type
document.querySelectorAll('input[name="order_type"]').forEach(radio => {
    radio.addEventListener('change', function() {
        const addressSection = document.getElementById('addressSection');
        const addressFields = addressSection.querySelectorAll('input[required], select[required]');
        
        if (this.value === 'pickup') {
            addressSection.style.display = 'none';
            addressFields.forEach(field => field.removeAttribute('required'));
        } else {
            addressSection.style.display = 'block';
            addressFields.forEach(field => field.setAttribute('required', 'required'));
        }
    });
});

// Payment method selection
document.querySelectorAll('.payment-method').forEach(method => {
    method.addEventListener('click', function() {
        document.querySelectorAll('.payment-method').forEach(m => m.classList.remove('active'));
        this.classList.add('active');
    });
});
</script>

<?php require_once 'includes/footer.php'; ?>

