<?php
/**
 * Create Order
 * POST /api/orders/create.php
 */

include_once '../config/cors.php';
include_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$data = json_decode(file_get_contents("php://input"));

if (!empty($data->customer_name) && !empty($data->customer_phone) && !empty($data->delivery_address) && !empty($data->items)) {
    try {
        $db->beginTransaction();
        
        // Generate order number
        $orderNumber = 'ORD' . date('Ymd') . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
        
        // Calculate totals
        $subtotal = 0;
        foreach ($data->items as $item) {
            $subtotal += $item->total_price;
        }
        
        $deliveryFee = isset($data->delivery_fee) ? $data->delivery_fee : 100;
        $tax = ($subtotal * 0.05); // 5% tax
        $discount = isset($data->discount) ? $data->discount : 0;
        $totalAmount = $subtotal + $deliveryFee + $tax - $discount;
        
        // Estimated delivery time (45 minutes from now)
        $estimatedDelivery = date('Y-m-d H:i:s', strtotime('+45 minutes'));
        
        // Prepare variables
        $userId = isset($data->user_id) ? $data->user_id : null;
        $customerName = $data->customer_name;
        $customerEmail = isset($data->customer_email) ? $data->customer_email : '';
        $customerPhone = $data->customer_phone;
        $deliveryAddress = $data->delivery_address;
        $paymentMethod = isset($data->payment_method) ? $data->payment_method : 'cash';
        $orderType = isset($data->order_type) ? $data->order_type : 'delivery';
        $specialInstructions = isset($data->special_instructions) ? $data->special_instructions : '';
        
        // Create order
        $query = "INSERT INTO orders 
                  (order_number, user_id, customer_name, customer_email, customer_phone, delivery_address, 
                   subtotal, delivery_fee, tax, discount, total_amount, payment_method, order_type, 
                   special_instructions, estimated_delivery_time) 
                  VALUES 
                  (:order_number, :user_id, :customer_name, :customer_email, :customer_phone, :delivery_address,
                   :subtotal, :delivery_fee, :tax, :discount, :total_amount, :payment_method, :order_type,
                   :special_instructions, :estimated_delivery_time)";
        
        $stmt = $db->prepare($query);
        
        $stmt->bindValue(':order_number', $orderNumber);
        $stmt->bindValue(':user_id', $userId);
        $stmt->bindValue(':customer_name', $customerName);
        $stmt->bindValue(':customer_email', $customerEmail);
        $stmt->bindValue(':customer_phone', $customerPhone);
        $stmt->bindValue(':delivery_address', $deliveryAddress);
        $stmt->bindValue(':subtotal', $subtotal);
        $stmt->bindValue(':delivery_fee', $deliveryFee);
        $stmt->bindValue(':tax', $tax);
        $stmt->bindValue(':discount', $discount);
        $stmt->bindValue(':total_amount', $totalAmount);
        $stmt->bindValue(':payment_method', $paymentMethod);
        $stmt->bindValue(':order_type', $orderType);
        $stmt->bindValue(':special_instructions', $specialInstructions);
        $stmt->bindValue(':estimated_delivery_time', $estimatedDelivery);
        
        $stmt->execute();
        $orderId = $db->lastInsertId();
        
        // Insert order items
        $itemQuery = "INSERT INTO order_items (order_id, menu_item_id, variant_id, quantity, unit_price, total_price, special_instructions) 
                      VALUES (:order_id, :menu_item_id, :variant_id, :quantity, :unit_price, :total_price, :special_instructions)";
        $itemStmt = $db->prepare($itemQuery);
        
        foreach ($data->items as $item) {
            $variantId = isset($item->variant_id) ? $item->variant_id : null;
            $itemInstructions = isset($item->special_instructions) ? $item->special_instructions : '';
            
            $itemStmt->bindValue(':order_id', $orderId);
            $itemStmt->bindValue(':menu_item_id', $item->menu_item_id);
            $itemStmt->bindValue(':variant_id', $variantId);
            $itemStmt->bindValue(':quantity', $item->quantity);
            $itemStmt->bindValue(':unit_price', $item->unit_price);
            $itemStmt->bindValue(':total_price', $item->total_price);
            $itemStmt->bindValue(':special_instructions', $itemInstructions);
            $itemStmt->execute();
            
            $orderItemId = $db->lastInsertId();
            
            // Insert addons if any
            if (!empty($item->addons) && is_array($item->addons)) {
                $addonQuery = "INSERT INTO order_item_addons (order_item_id, addon_id, quantity, price) VALUES (:order_item_id, :addon_id, :quantity, :price)";
                $addonStmt = $db->prepare($addonQuery);
                
                foreach ($item->addons as $addon) {
                    $addonQty = isset($addon->quantity) ? $addon->quantity : 1;
                    
                    $addonStmt->bindValue(':order_item_id', $orderItemId);
                    $addonStmt->bindValue(':addon_id', $addon->addon_id);
                    $addonStmt->bindValue(':quantity', $addonQty);
                    $addonStmt->bindValue(':price', $addon->price);
                    $addonStmt->execute();
                }
            }
        }
        
        $db->commit();
        
        http_response_code(201);
        echo json_encode([
            "success" => true,
            "message" => "Order placed successfully",
            "order_number" => $orderNumber,
            "order_id" => $orderId,
            "total_amount" => $totalAmount,
            "estimated_delivery" => $estimatedDelivery
        ]);
    } catch(PDOException $e) {
        $db->rollBack();
        http_response_code(500);
        echo json_encode([
            "success" => false,
            "message" => "Error creating order: " . $e->getMessage()
        ]);
    } catch(Exception $e) {
        $db->rollBack();
        http_response_code(500);
        echo json_encode([
            "success" => false,
            "message" => "Error: " . $e->getMessage()
        ]);
    }
} else {
    http_response_code(400);
    echo json_encode([
        "success" => false,
        "message" => "Customer name, phone, address and items are required"
    ]);
}
