<?php
/**
 * Create Menu Item
 * POST /api/menu/create.php
 */

include_once '../config/cors.php';
include_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$data = json_decode(file_get_contents("php://input"));

// Debug logging (remove in production)
error_log("Menu Create Request: " . json_encode($data));

if (!empty($data->name) && !empty($data->category_id) && !empty($data->price)) {
    try {
        $db->beginTransaction();
        
        $query = "INSERT INTO menu_items 
                  (category_id, name, description, price, discount_price, image, is_vegetarian, is_spicy, is_featured, is_available, preparation_time, calories) 
                  VALUES 
                  (:category_id, :name, :description, :price, :discount_price, :image, :is_vegetarian, :is_spicy, :is_featured, :is_available, :preparation_time, :calories)";
        
        $stmt = $db->prepare($query);
        
        // Convert boolean values properly
        $is_vegetarian = isset($data->is_vegetarian) ? (bool)$data->is_vegetarian : false;
        $is_spicy = isset($data->is_spicy) ? (bool)$data->is_spicy : false;
        $is_featured = isset($data->is_featured) ? (bool)$data->is_featured : false;
        $is_available = isset($data->is_available) ? (bool)$data->is_available : true;
        
        // Convert to integers for MySQL (MySQL uses TINYINT for BOOLEAN)
        $is_vegetarian_int = $is_vegetarian ? 1 : 0;
        $is_spicy_int = $is_spicy ? 1 : 0;
        $is_featured_int = $is_featured ? 1 : 0;
        $is_available_int = $is_available ? 1 : 0;
        
        $prep_time = isset($data->preparation_time) ? (int)$data->preparation_time : 30;
        $calories_val = isset($data->calories) && $data->calories !== '' ? (int)$data->calories : null;
        $discount_price_val = isset($data->discount_price) && $data->discount_price !== '' && $data->discount_price !== null ? (float)$data->discount_price : null;
        
        // Prepare variables for bindValue (bindValue works better with NULL values)
        $category_id = (int)$data->category_id;
        $name = $data->name;
        $description = isset($data->description) ? $data->description : '';
        $price = (float)$data->price;
        $image = isset($data->image) ? $data->image : '';
        
        $stmt->bindValue(':category_id', $category_id, PDO::PARAM_INT);
        $stmt->bindValue(':name', $name, PDO::PARAM_STR);
        $stmt->bindValue(':description', $description, PDO::PARAM_STR);
        $stmt->bindValue(':price', $price);
        $stmt->bindValue(':discount_price', $discount_price_val);
        $stmt->bindValue(':image', $image, PDO::PARAM_STR);
        $stmt->bindValue(':is_vegetarian', $is_vegetarian_int, PDO::PARAM_INT);
        $stmt->bindValue(':is_spicy', $is_spicy_int, PDO::PARAM_INT);
        $stmt->bindValue(':is_featured', $is_featured_int, PDO::PARAM_INT);
        $stmt->bindValue(':is_available', $is_available_int, PDO::PARAM_INT);
        $stmt->bindValue(':preparation_time', $prep_time, PDO::PARAM_INT);
        $stmt->bindValue(':calories', $calories_val);
        
        $stmt->execute();
        $itemId = $db->lastInsertId();
        
        // Add variants if provided
        if (!empty($data->variants) && is_array($data->variants)) {
            $variantQuery = "INSERT INTO item_variants (menu_item_id, name, price_addition, is_default) VALUES (:menu_item_id, :name, :price_addition, :is_default)";
            $variantStmt = $db->prepare($variantQuery);
            
            foreach ($data->variants as $variant) {
                $variantStmt->bindParam(':menu_item_id', $itemId);
                $variantStmt->bindParam(':name', $variant->name);
                $variantStmt->bindParam(':price_addition', $variant->price_addition ?? 0);
                $variantStmt->bindParam(':is_default', $variant->is_default ?? false);
                $variantStmt->execute();
            }
        }
        
        $db->commit();
        
        http_response_code(200);
        echo json_encode([
            "success" => true,
            "message" => "Menu item created successfully",
            "id" => $itemId,
            "data" => [
                "id" => $itemId,
                "name" => $data->name,
                "category_id" => $data->category_id
            ]
        ], JSON_UNESCAPED_UNICODE);
    } catch(PDOException $e) {
        $db->rollBack();
        error_log("Menu Create Error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode([
            "success" => false,
            "message" => "Error creating menu item: " . $e->getMessage(),
            "error_details" => $e->getMessage()
        ]);
    }
} else {
    http_response_code(400);
    echo json_encode([
        "success" => false,
        "message" => "Name, category_id and price are required"
    ]);
}

