<?php
/**
 * User Registration
 * POST /api/auth/register.php
 */

include_once '../config/cors.php';
include_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$data = json_decode(file_get_contents("php://input"));

if (!empty($data->name) && !empty($data->email) && !empty($data->password) && !empty($data->phone)) {
    // Validate email
    if (!filter_var($data->email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        echo json_encode([
            "success" => false,
            "message" => "Invalid email format"
        ]);
        exit;
    }
    
    // Validate password length
    if (strlen($data->password) < 6) {
        http_response_code(400);
        echo json_encode([
            "success" => false,
            "message" => "Password must be at least 6 characters"
        ]);
        exit;
    }
    
    try {
        // Check if email already exists
        $checkQuery = "SELECT id FROM users WHERE email = :email";
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':email', $data->email);
        $checkStmt->execute();
        
        if ($checkStmt->rowCount() > 0) {
            http_response_code(409);
            echo json_encode([
                "success" => false,
                "message" => "Email already registered"
            ]);
            exit;
        }
        
        // Hash password
        $hashedPassword = password_hash($data->password, PASSWORD_DEFAULT);
        
        $query = "INSERT INTO users (name, email, password, phone, address, role) VALUES (:name, :email, :password, :phone, :address, 'customer')";
        $stmt = $db->prepare($query);
        
        $stmt->bindParam(':name', $data->name);
        $stmt->bindParam(':email', $data->email);
        $stmt->bindParam(':password', $hashedPassword);
        $stmt->bindParam(':phone', $data->phone);
        $stmt->bindParam(':address', $data->address ?? '');
        
        if ($stmt->execute()) {
            $userId = $db->lastInsertId();
            
            // Generate token
            $token = bin2hex(random_bytes(32));
            
            http_response_code(201);
            echo json_encode([
                "success" => true,
                "message" => "Registration successful",
                "token" => $token,
                "user" => [
                    "id" => $userId,
                    "name" => $data->name,
                    "email" => $data->email,
                    "phone" => $data->phone,
                    "role" => "customer"
                ]
            ]);
        }
    } catch(PDOException $e) {
        http_response_code(500);
        echo json_encode([
            "success" => false,
            "message" => "Error: " . $e->getMessage()
        ]);
    }
} else {
    http_response_code(400);
    echo json_encode([
        "success" => false,
        "message" => "Name, email, password and phone are required"
    ]);
}

