<?php
require_once 'config.php';
requireAdminLogin();

$pageTitle = 'Orders';
require_once 'includes/header.php';
?>

<div class="page-title">
    <div>
        <h1>Orders</h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item active">Orders</li>
            </ol>
        </nav>
    </div>
</div>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-body">
        <div class="row g-3">
            <div class="col-md-3">
                <input type="text" class="form-control" id="searchOrder" placeholder="Search orders...">
            </div>
            <div class="col-md-3">
                <select class="form-select" id="statusFilter">
                    <option value="">All Status</option>
                    <option value="pending">Pending</option>
                    <option value="confirmed">Confirmed</option>
                    <option value="preparing">Preparing</option>
                    <option value="ready">Ready</option>
                    <option value="out_for_delivery">Out for Delivery</option>
                    <option value="delivered">Delivered</option>
                </select>
            </div>
        </div>
    </div>
</div>

<!-- Orders Table -->
<div class="card">
    <div class="card-body p-0">
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead>
                    <tr>
                        <th>Order ID</th>
                        <th>Customer</th>
                        <th>Items</th>
                        <th>Total</th>
                        <th>Status</th>
                        <th>Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody id="ordersTableBody">
                    <tr>
                        <td colspan="7" class="text-center py-4">
                            <div class="loading"></div> Loading orders...
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
$(document).ready(function() {
    loadOrders();
    
    $('#searchOrder, #statusFilter').on('input change', function() {
        loadOrders();
    });
});

function loadOrders() {
    const params = {
        search: $('#searchOrder').val(),
        status: $('#statusFilter').val()
    };
    
    AdminAPI.get('/orders/read.php', params)
        .done(function(response) {
            if (response.success) {
                displayOrders(response.data);
            }
        })
        .fail(function(xhr) {
            handleApiError(xhr);
        });
}

function displayOrders(orders) {
    const tbody = $('#ordersTableBody');
    tbody.empty();
    
    if (orders.length === 0) {
        tbody.html('<tr><td colspan="7" class="text-center py-4 text-muted">No orders found</td></tr>');
        return;
    }
    
    orders.forEach(function(order) {
        const row = $(`
            <tr>
                <td><strong>${order.order_number}</strong></td>
                <td>${order.customer_name}</td>
                <td>${order.items ? order.items.length : 0} items</td>
                <td>${formatPrice(order.total_amount)}</td>
                <td>
                    <select class="form-select form-select-sm status-select" data-order-id="${order.id}" onchange="updateOrderStatus(${order.id}, this.value)">
                        <option value="pending" ${order.order_status === 'pending' ? 'selected' : ''}>Pending</option>
                        <option value="confirmed" ${order.order_status === 'confirmed' ? 'selected' : ''}>Confirmed</option>
                        <option value="preparing" ${order.order_status === 'preparing' ? 'selected' : ''}>Preparing</option>
                        <option value="ready" ${order.order_status === 'ready' ? 'selected' : ''}>Ready</option>
                        <option value="out_for_delivery" ${order.order_status === 'out_for_delivery' ? 'selected' : ''}>Out for Delivery</option>
                        <option value="delivered" ${order.order_status === 'delivered' ? 'selected' : ''}>Delivered</option>
                    </select>
                </td>
                <td>${formatDate(order.created_at)}</td>
                <td>
                    <button class="btn btn-sm btn-outline-primary" onclick="viewOrder(${order.id})">
                        <i class="bi bi-eye"></i>
                    </button>
                </td>
            </tr>
        `);
        tbody.append(row);
    });
}

function updateOrderStatus(orderId, status) {
    AdminAPI.put('/orders/update_status.php', {
        order_id: orderId,
        status: status
    })
    .done(function(response) {
        if (response.success) {
            showNotification('success', 'Order status updated successfully');
            loadOrders();
        } else {
            showNotification('error', response.message || 'Failed to update status');
        }
    })
    .fail(function(xhr) {
        handleApiError(xhr);
    });
}

function viewOrder(id) {
    // Open order details modal or page
    window.location.href = 'order-details.php?id=' + id;
}
</script>

<?php require_once 'includes/footer.php'; ?>

