<?php
require_once 'config.php';
requireAdminLogin();

$pageTitle = 'Menu Items';
require_once 'includes/header.php';

// Get categories for dropdown
$db = getDB();
$categories = $db->query("SELECT * FROM categories ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
?>

<div class="page-title">
    <div>
        <h1>Menu Items</h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item active">Menu Items</li>
            </ol>
        </nav>
    </div>
    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#itemModal" onclick="openAddModal()">
        <i class="bi bi-plus-lg me-2"></i>Add Item
    </button>
</div>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-body">
        <div class="row g-3 align-items-center">
            <div class="col-md-4">
                <input type="text" class="form-control" id="searchQuery" placeholder="Search menu items...">
            </div>
            <div class="col-md-3">
                <select class="form-select" id="categoryFilter">
                    <option value="">All Categories</option>
                    <?php foreach ($categories as $cat): ?>
                        <option value="<?= $cat['id'] ?>"><?= htmlspecialchars($cat['name']) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-2">
                <select class="form-select" id="availabilityFilter">
                    <option value="">All Status</option>
                    <option value="1">Available</option>
                    <option value="0">Unavailable</option>
                </select>
            </div>
        </div>
    </div>
</div>

<!-- Menu Items Grid -->
<div class="row g-4" id="menuItemsContainer">
    <!-- Items will be loaded here via AJAX -->
</div>

<!-- Add/Edit Modal -->
<div class="modal fade" id="itemModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="modalTitle">Add Menu Item</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="itemForm">
                    <input type="hidden" id="itemId">
                    <div class="row g-3">
                        <div class="col-md-8">
                            <label class="form-label">Item Name *</label>
                            <input type="text" class="form-control" id="itemName" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Category *</label>
                            <select class="form-select" id="itemCategory" required>
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $cat): ?>
                                    <option value="<?= $cat['id'] ?>"><?= htmlspecialchars($cat['name']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" rows="3" id="itemDescription"></textarea>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Price ($) *</label>
                            <input type="number" class="form-control" id="itemPrice" step="0.01" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Discount Price ($)</label>
                            <input type="number" class="form-control" id="itemDiscountPrice" step="0.01">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Prep Time (mins)</label>
                            <input type="number" class="form-control" id="itemPrepTime" value="30">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Image URL or Filename</label>
                            <input type="text" class="form-control" id="itemImage" placeholder="pizza.jpg or https://example.com/image.jpg">
                            <small class="text-muted">Enter filename (e.g., pizza.jpg) or full URL</small>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Calories</label>
                            <input type="number" class="form-control" id="itemCalories">
                        </div>
                        <div class="col-12">
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="checkbox" id="itemAvailable" checked>
                                <label class="form-check-label" for="itemAvailable">Available</label>
                            </div>
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="checkbox" id="itemFeatured">
                                <label class="form-check-label" for="itemFeatured">Featured</label>
                            </div>
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="checkbox" id="itemVegetarian">
                                <label class="form-check-label" for="itemVegetarian">Vegetarian</label>
                            </div>
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="checkbox" id="itemSpicy">
                                <label class="form-check-label" for="itemSpicy">Spicy</label>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="saveItem()">
                    <i class="bi bi-check-lg me-2"></i>Save Item
                </button>
            </div>
        </div>
    </div>
</div>

<script>
let editingItem = null;

$(document).ready(function() {
    loadMenuItems();
    
    $('#searchQuery, #categoryFilter, #availabilityFilter').on('input change', function() {
        loadMenuItems();
    });
});

function loadMenuItems() {
    const params = {
        search: $('#searchQuery').val(),
        category_id: $('#categoryFilter').val(),
        is_available: $('#availabilityFilter').val()
    };
    
    AdminAPI.get('/menu/read.php', params)
        .done(function(response) {
            if (response.success) {
                displayMenuItems(response.data);
            }
        })
        .fail(function(xhr) {
            handleApiError(xhr);
        });
}

function displayMenuItems(items) {
    const container = $('#menuItemsContainer');
    container.empty();
    
    if (items.length === 0) {
        container.html('<div class="col-12 text-center py-5"><i class="bi bi-egg-fried text-muted" style="font-size: 64px;"></i><p class="text-muted mt-3">No menu items found</p></div>');
        return;
    }
    
    items.forEach(function(item) {
        const imageUrl = getImageUrl(item.image);
        const card = $(`
            <div class="col-lg-3 col-md-4 col-sm-6">
                <div class="card h-100">
                    <div class="position-relative">
                        <img src="${imageUrl}" class="card-img-top" style="height: 180px; object-fit: cover;" alt="${item.name}" onerror="this.src='https://images.unsplash.com/photo-1565299624946-b28f40a0ae38?w=400&h=250&fit=crop'">
                        <div class="position-absolute top-0 end-0 m-2">
                            <span class="badge ${item.is_available ? 'bg-success' : 'bg-danger'}">
                                ${item.is_available ? 'Available' : 'Unavailable'}
                            </span>
                        </div>
                    </div>
                    <div class="card-body">
                        <small class="text-primary text-uppercase">${item.category_name || 'Uncategorized'}</small>
                        <h6 class="card-title mt-1 mb-2">${item.name}</h6>
                        <p class="card-text text-muted small mb-2" style="height: 40px; overflow: hidden;">${item.description || ''}</p>
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <strong class="text-primary">$${parseFloat(item.discount_price || item.price).toFixed(2)}</strong>
                                ${item.discount_price ? `<small class="text-muted text-decoration-line-through ms-1">$${parseFloat(item.price).toFixed(2)}</small>` : ''}
                            </div>
                            <div class="btn-group btn-group-sm">
                                <button class="btn btn-outline-primary" onclick="editItem(${item.id})" title="Edit">
                                    <i class="bi bi-pencil"></i>
                                </button>
                                <button class="btn btn-outline-danger" onclick="deleteItem(${item.id})" title="Delete">
                                    <i class="bi bi-trash"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `);
        container.append(card);
    });
}

function getImageUrl(image) {
    if (!image || image.trim() === '') {
        return 'https://images.unsplash.com/photo-1565299624946-b28f40a0ae38?w=400&h=250&fit=crop';
    }
    const img = image.trim();
    if (img.match(/^https?:\/\//i)) {
        return img;
    }
    if (img.toLowerCase().startsWith('uploads/')) {
        return 'http://localhost/orderfood/frontend/' + img;
    }
    return 'http://localhost/orderfood/frontend/uploads/' + img;
}

function openAddModal() {
    editingItem = null;
    $('#modalTitle').text('Add Menu Item');
    $('#itemForm')[0].reset();
    $('#itemId').val('');
    $('#itemAvailable').prop('checked', true);
}

function editItem(id) {
    AdminAPI.get('/menu/read.php', { id: id })
        .done(function(response) {
            if (response.success && response.data.length > 0) {
                const item = response.data[0];
                editingItem = item;
                $('#modalTitle').text('Edit Menu Item');
                $('#itemId').val(item.id);
                $('#itemName').val(item.name);
                $('#itemCategory').val(item.category_id);
                $('#itemDescription').val(item.description || '');
                $('#itemPrice').val(item.price);
                $('#itemDiscountPrice').val(item.discount_price || '');
                $('#itemPrepTime').val(item.preparation_time || 30);
                $('#itemImage').val(item.image || '');
                $('#itemCalories').val(item.calories || '');
                $('#itemAvailable').prop('checked', item.is_available == 1);
                $('#itemFeatured').prop('checked', item.is_featured == 1);
                $('#itemVegetarian').prop('checked', item.is_vegetarian == 1);
                $('#itemSpicy').prop('checked', item.is_spicy == 1);
                
                const modal = new bootstrap.Modal(document.getElementById('itemModal'));
                modal.show();
            }
        })
        .fail(function(xhr) {
            handleApiError(xhr);
        });
}

function saveItem() {
    const data = {
        name: $('#itemName').val(),
        category_id: $('#itemCategory').val(),
        description: $('#itemDescription').val(),
        price: $('#itemPrice').val(),
        discount_price: $('#itemDiscountPrice').val() || null,
        image: $('#itemImage').val() || '',
        preparation_time: $('#itemPrepTime').val() || 30,
        calories: $('#itemCalories').val() || null,
        is_available: $('#itemAvailable').is(':checked') ? 1 : 0,
        is_featured: $('#itemFeatured').is(':checked') ? 1 : 0,
        is_vegetarian: $('#itemVegetarian').is(':checked') ? 1 : 0,
        is_spicy: $('#itemSpicy').is(':checked') ? 1 : 0
    };
    
    if (!data.name || !data.category_id || !data.price) {
        showNotification('error', 'Please fill in all required fields');
        return;
    }
    
    const button = event.target;
    setLoading(button, true);
    
    if (editingItem) {
        // Update (if API exists)
        showNotification('warning', 'Update functionality will be added soon');
        setLoading(button, false);
    } else {
        // Create
        AdminAPI.post('/menu/create.php', data)
            .done(function(response) {
                if (response.success) {
                    showNotification('success', 'Menu item created successfully');
                    $('#itemModal').modal('hide');
                    loadMenuItems();
                } else {
                    showNotification('error', response.message || 'Failed to create item');
                }
                setLoading(button, false);
            })
            .fail(function(xhr) {
                handleApiError(xhr);
                setLoading(button, false);
            });
    }
}

function deleteItem(id) {
    if (!confirmDelete('Are you sure you want to delete this menu item?')) {
        return;
    }
    
    AdminAPI.delete('/menu/delete.php', { id: id })
        .done(function(response) {
            if (response.success) {
                showNotification('success', 'Menu item deleted successfully');
                loadMenuItems();
            } else {
                showNotification('error', response.message || 'Failed to delete item');
            }
        })
        .fail(function(xhr) {
            handleApiError(xhr);
        });
}
</script>

<?php require_once 'includes/footer.php'; ?>

