<?php
require_once 'config.php';
requireAdminLogin();

$pageTitle = 'Branch Locations';
require_once 'includes/header.php';
?>

<div class="page-title">
    <div>
        <h1>Branch Locations</h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item active">Locations</li>
            </ol>
        </nav>
    </div>
    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#locationModal" onclick="openAddModal()">
        <i class="bi bi-plus-lg me-2"></i>Add Location
    </button>
</div>

<div class="row g-4" id="locationsContainer">
    <!-- Locations will be loaded here -->
</div>

<!-- Add/Edit Modal -->
<div class="modal fade" id="locationModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="modalTitle">Add Location</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="locationForm">
                    <input type="hidden" id="locationId">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Location Name *</label>
                            <input type="text" class="form-control" id="locationName" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Phone</label>
                            <input type="text" class="form-control" id="locationPhone">
                        </div>
                        <div class="col-12">
                            <label class="form-label">Address</label>
                            <textarea class="form-control" rows="2" id="locationAddress"></textarea>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">City</label>
                            <input type="text" class="form-control" id="locationCity">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">State</label>
                            <input type="text" class="form-control" id="locationState">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Zip Code</label>
                            <input type="text" class="form-control" id="locationZip">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Country</label>
                            <input type="text" class="form-control" id="locationCountry">
                        </div>
                        <div class="col-12">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="locationActive" checked>
                                <label class="form-check-label" for="locationActive">Active</label>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="saveLocation()">Save Location</button>
            </div>
        </div>
    </div>
</div>

<script>
let editingLocation = null;

$(document).ready(function() {
    loadLocations();
});

function loadLocations() {
    AdminAPI.get('/locations/read.php')
        .done(function(response) {
            if (response.success) {
                displayLocations(response.data);
            }
        })
        .fail(function(xhr) {
            handleApiError(xhr);
        });
}

function displayLocations(locations) {
    const container = $('#locationsContainer');
    container.empty();
    
    if (locations.length === 0) {
        container.html('<div class="col-12 text-center py-5"><p class="text-muted">No locations found</p></div>');
        return;
    }
    
    locations.forEach(function(loc) {
        const card = $(`
            <div class="col-md-6">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">${loc.name}</h5>
                        <p class="card-text text-muted mb-2">
                            <i class="bi bi-geo-alt me-2"></i>${loc.address || 'No address'}
                        </p>
                        ${loc.phone ? `<p class="card-text text-muted mb-2"><i class="bi bi-telephone me-2"></i>${loc.phone}</p>` : ''}
                        <div class="d-flex justify-content-between align-items-center mt-3">
                            <span class="badge ${loc.is_active ? 'bg-success' : 'bg-secondary'}">${loc.is_active ? 'Active' : 'Inactive'}</span>
                            <div>
                                <button class="btn btn-sm btn-outline-primary" onclick="editLocation(${loc.id})">
                                    <i class="bi bi-pencil"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-danger" onclick="deleteLocation(${loc.id})">
                                    <i class="bi bi-trash"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `);
        container.append(card);
    });
}

function openAddModal() {
    editingLocation = null;
    $('#modalTitle').text('Add Location');
    $('#locationForm')[0].reset();
    $('#locationId').val('');
    $('#locationActive').prop('checked', true);
}

function editLocation(id) {
    AdminAPI.get('/locations/read.php', { id: id })
        .done(function(response) {
            if (response.success && response.data.length > 0) {
                const loc = response.data[0];
                editingLocation = loc;
                $('#modalTitle').text('Edit Location');
                $('#locationId').val(loc.id);
                $('#locationName').val(loc.name);
                $('#locationPhone').val(loc.phone || '');
                $('#locationAddress').val(loc.address || '');
                $('#locationCity').val(loc.city || '');
                $('#locationState').val(loc.state || '');
                $('#locationZip').val(loc.zip_code || '');
                $('#locationCountry').val(loc.country || '');
                $('#locationActive').prop('checked', loc.is_active == 1);
                $('#locationModal').modal('show');
            }
        })
        .fail(function(xhr) {
            handleApiError(xhr);
        });
}

function saveLocation() {
    const data = {
        name: $('#locationName').val(),
        phone: $('#locationPhone').val() || null,
        address: $('#locationAddress').val() || null,
        city: $('#locationCity').val() || null,
        state: $('#locationState').val() || null,
        zip_code: $('#locationZip').val() || null,
        country: $('#locationCountry').val() || null,
        is_active: $('#locationActive').is(':checked') ? 1 : 0
    };
    
    if (!data.name) {
        showNotification('error', 'Please enter location name');
        return;
    }
    
    const button = event.target;
    setLoading(button, true);
    
    if (editingLocation) {
        AdminAPI.put('/locations/update.php', { id: editingLocation.id, ...data })
            .done(function(response) {
                if (response.success) {
                    showNotification('success', 'Location updated successfully');
                    $('#locationModal').modal('hide');
                    loadLocations();
                } else {
                    showNotification('error', response.message || 'Failed to update location');
                }
                setLoading(button, false);
            })
            .fail(function(xhr) {
                handleApiError(xhr);
                setLoading(button, false);
            });
    } else {
        AdminAPI.post('/locations/create.php', data)
            .done(function(response) {
                if (response.success) {
                    showNotification('success', 'Location created successfully');
                    $('#locationModal').modal('hide');
                    loadLocations();
                } else {
                    showNotification('error', response.message || 'Failed to create location');
                }
                setLoading(button, false);
            })
            .fail(function(xhr) {
                handleApiError(xhr);
                setLoading(button, false);
            });
    }
}

function deleteLocation(id) {
    if (!confirmDelete('Are you sure you want to delete this location?')) {
        return;
    }
    
    AdminAPI.delete('/locations/delete.php', { id: id })
        .done(function(response) {
            if (response.success) {
                showNotification('success', 'Location deleted successfully');
                loadLocations();
            } else {
                showNotification('error', response.message || 'Failed to delete location');
            }
        })
        .fail(function(xhr) {
            handleApiError(xhr);
        });
}
</script>

<?php require_once 'includes/footer.php'; ?>

