<?php
require_once 'config.php';
requireAdminLogin();

$pageTitle = 'Categories';
require_once 'includes/header.php';
?>

<div class="page-title">
    <div>
        <h1>Categories</h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item active">Categories</li>
            </ol>
        </nav>
    </div>
    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#categoryModal" onclick="openAddModal()">
        <i class="bi bi-plus-lg me-2"></i>Add Category
    </button>
</div>

<div class="row g-4" id="categoriesContainer">
    <!-- Categories will be loaded here -->
</div>

<!-- Add/Edit Modal -->
<div class="modal fade" id="categoryModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="modalTitle">Add Category</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="categoryForm">
                    <input type="hidden" id="categoryId">
                    <div class="mb-3">
                        <label class="form-label">Category Name *</label>
                        <input type="text" class="form-control" id="categoryName" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea class="form-control" rows="3" id="categoryDescription"></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Sort Order</label>
                        <input type="number" class="form-control" id="categorySortOrder" value="0">
                    </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="categoryActive" checked>
                        <label class="form-check-label" for="categoryActive">Active</label>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="saveCategory()">Save Category</button>
            </div>
        </div>
    </div>
</div>

<script>
let editingCategory = null;

$(document).ready(function() {
    loadCategories();
});

function loadCategories() {
    AdminAPI.get('/categories/read.php')
        .done(function(response) {
            if (response.success) {
                displayCategories(response.data);
            }
        })
        .fail(function(xhr) {
            handleApiError(xhr);
        });
}

function displayCategories(categories) {
    const container = $('#categoriesContainer');
    container.empty();
    
    if (categories.length === 0) {
        container.html('<div class="col-12 text-center py-5"><p class="text-muted">No categories found</p></div>');
        return;
    }
    
    categories.forEach(function(cat) {
        const card = $(`
            <div class="col-md-4">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">${cat.name}</h5>
                        <p class="card-text text-muted">${cat.description || 'No description'}</p>
                        <div class="d-flex justify-content-between align-items-center">
                            <span class="badge ${cat.is_active ? 'bg-success' : 'bg-secondary'}">${cat.is_active ? 'Active' : 'Inactive'}</span>
                            <div>
                                <button class="btn btn-sm btn-outline-primary" onclick="editCategory(${cat.id})">
                                    <i class="bi bi-pencil"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-danger" onclick="deleteCategory(${cat.id})">
                                    <i class="bi bi-trash"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `);
        container.append(card);
    });
}

function openAddModal() {
    editingCategory = null;
    $('#modalTitle').text('Add Category');
    $('#categoryForm')[0].reset();
    $('#categoryId').val('');
    $('#categoryActive').prop('checked', true);
}

function editCategory(id) {
    AdminAPI.get('/categories/read.php', { id: id })
        .done(function(response) {
            if (response.success && response.data.length > 0) {
                const cat = response.data[0];
                editingCategory = cat;
                $('#modalTitle').text('Edit Category');
                $('#categoryId').val(cat.id);
                $('#categoryName').val(cat.name);
                $('#categoryDescription').val(cat.description || '');
                $('#categorySortOrder').val(cat.sort_order || 0);
                $('#categoryActive').prop('checked', cat.is_active == 1);
                $('#categoryModal').modal('show');
            }
        })
        .fail(function(xhr) {
            handleApiError(xhr);
        });
}

function saveCategory() {
    const data = {
        name: $('#categoryName').val(),
        description: $('#categoryDescription').val() || '',
        sort_order: $('#categorySortOrder').val() || 0,
        is_active: $('#categoryActive').is(':checked') ? 1 : 0
    };
    
    if (!data.name) {
        showNotification('error', 'Please enter category name');
        return;
    }
    
    const button = event.target;
    setLoading(button, true);
    
    if (editingCategory) {
        AdminAPI.put('/categories/update.php', { id: editingCategory.id, ...data })
            .done(function(response) {
                if (response.success) {
                    showNotification('success', 'Category updated successfully');
                    $('#categoryModal').modal('hide');
                    loadCategories();
                } else {
                    showNotification('error', response.message || 'Failed to update category');
                }
                setLoading(button, false);
            })
            .fail(function(xhr) {
                handleApiError(xhr);
                setLoading(button, false);
            });
    } else {
        AdminAPI.post('/categories/create.php', data)
            .done(function(response) {
                if (response.success) {
                    showNotification('success', 'Category created successfully');
                    $('#categoryModal').modal('hide');
                    loadCategories();
                } else {
                    showNotification('error', response.message || 'Failed to create category');
                }
                setLoading(button, false);
            })
            .fail(function(xhr) {
                handleApiError(xhr);
                setLoading(button, false);
            });
    }
}

function deleteCategory(id) {
    if (!confirmDelete('Are you sure you want to delete this category?')) {
        return;
    }
    
    AdminAPI.delete('/categories/delete.php', { id: id })
        .done(function(response) {
            if (response.success) {
                showNotification('success', 'Category deleted successfully');
                loadCategories();
            } else {
                showNotification('error', response.message || 'Failed to delete category');
            }
        })
        .fail(function(xhr) {
            handleApiError(xhr);
        });
}
</script>

<?php require_once 'includes/footer.php'; ?>

