/**
 * OrderFood Admin Panel - JavaScript
 */

const API_URL = 'http://localhost/orderfood/api';

// Sidebar Toggle
$(document).ready(function() {
    // Toggle sidebar
    $('#toggleSidebar').on('click', function() {
        $('#sidebar').toggleClass('collapsed');
    });
    
    // Mobile sidebar toggle
    if ($(window).width() <= 768) {
        $('#toggleSidebar').on('click', function() {
            $('#sidebar').toggleClass('show');
        });
    }
});

// API Helper Functions
const AdminAPI = {
    // GET request
    get: function(endpoint, params = {}) {
        let url = API_URL + endpoint;
        if (Object.keys(params).length > 0) {
            url += '?' + $.param(params);
        }
        return $.ajax({
            url: url,
            method: 'GET',
            dataType: 'json'
        });
    },
    
    // POST request
    post: function(endpoint, data) {
        return $.ajax({
            url: API_URL + endpoint,
            method: 'POST',
            contentType: 'application/json',
            data: JSON.stringify(data),
            dataType: 'json'
        });
    },
    
    // PUT request
    put: function(endpoint, data) {
        return $.ajax({
            url: API_URL + endpoint,
            method: 'PUT',
            contentType: 'application/json',
            data: JSON.stringify(data),
            dataType: 'json'
        });
    },
    
    // DELETE request
    delete: function(endpoint, params = {}) {
        let url = API_URL + endpoint;
        if (Object.keys(params).length > 0) {
            url += '?' + $.param(params);
        }
        return $.ajax({
            url: url,
            method: 'DELETE',
            dataType: 'json'
        });
    }
};

// Notification System
function showNotification(type, message) {
    const alertClass = type === 'success' ? 'alert-success' : 
                      type === 'error' ? 'alert-danger' : 
                      type === 'warning' ? 'alert-warning' : 'alert-info';
    
    const notification = $(`
        <div class="alert ${alertClass} alert-dismissible fade show position-fixed top-0 end-0 m-3" style="z-index: 9999; min-width: 300px;" role="alert">
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    `);
    
    $('body').append(notification);
    
    setTimeout(function() {
        notification.fadeOut(function() {
            $(this).remove();
        });
    }, 3000);
}

// Format Price
function formatPrice(price) {
    return '$' + parseFloat(price).toLocaleString('en-US', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
}

// Format Date
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// Confirm Delete
function confirmDelete(message = 'Are you sure you want to delete this item?') {
    return confirm(message);
}

// Loading Button
function setLoading(button, loading) {
    if (loading) {
        $(button).prop('disabled', true);
        $(button).data('original-text', $(button).html());
        $(button).html('<span class="loading"></span> Loading...');
    } else {
        $(button).prop('disabled', false);
        $(button).html($(button).data('original-text'));
    }
}

// Handle API Errors
function handleApiError(xhr) {
    let message = 'An error occurred';
    
    if (xhr.responseJSON && xhr.responseJSON.message) {
        message = xhr.responseJSON.message;
    } else if (xhr.status === 0) {
        message = 'Network error. Please check your connection.';
    } else if (xhr.status === 401) {
        message = 'Unauthorized. Please login again.';
        setTimeout(function() {
            window.location.href = 'login.php';
        }, 2000);
    } else if (xhr.status === 500) {
        message = 'Server error. Please try again later.';
    }
    
    showNotification('error', message);
}

